// ==============================================================
//                 ORBITER MODULE:  ShuttleA2
//			Modification by Alexander Blass 2003 - 2004
//			based on ShuttleA (C) 2002 Martin Schweiger
//
// ShuttleA.cpp
// Reference implementation of Shuttle-A vessel class module
// ==============================================================

#define STRICT

#include "MovableThruster.h"
#include "ShuttleA.h"
#include "UpperPanel.h"
#include "MainPanel.h"
#include "resource.h"
#include <math.h>
#include <stdio.h>
#include "orbitersoundsdk.h"

#define LOADBMP(id) (LoadBitmap (g_Param.hDLL, MAKEINTRESOURCE (id)))

// ==============================================================
// Global parameters
// ==============================================================

GDIParams g_Param;
int g_iSoundBase = 0;

// ==============================================================
// Specialised vessel class ShuttleA
// ==============================================================

// Constructor
ShuttleA::ShuttleA (OBJHANDLE hObj, int fmodel)
: VESSEL (hObj, fmodel)
{
	int i;
	m_bShuttleBroken = false;
	dock_proc = 0.0;
	dock_status = DOOR_CLOSED;
	lock_proc = 0.0;
	lock_status = DOOR_CLOSED;
	m_bRadDmgAlarm = m_bShuttleBroken = m_bOvrSpdAlarm= m_bHullTempAlarm = m_bDmgAlarm = m_bAuxPodDmg = false;
	m_uiCurPodsGroup = USER;
	m_vh = NULL;
	m_bThrDmgAdded = false;
	m_arrpAuxThrusters[0] = new CMovableThruster(this, _V(-7.0 ,0.0, 0.0), MAX_RETRO_THRUST, ISP, _V(0.0, 0.0, -1.0), 0.0, PI, CMovableThruster::X, 6, 0.7, 1.5);
	m_arrpAuxThrusters[1] = new CMovableThruster(this, _V(7.0 ,0.0, 0.0), MAX_RETRO_THRUST, ISP, _V(0.0, 0.0, -1.0), 0.0, PI, CMovableThruster::X, 6, 0.7, 1.5);
	m_pRadar = new CRadar(this);
	m_iNumExtTanks = 0;
	for (i = 0; i < 4; i++)
	{
		m_arriCargoConf[i] = EMPTY;
		m_arrdCargoWeights[i] = 3600;
		m_arrMeshesCargoSlots[i] = new char[256];
		m_arrMeshesCargoSlots[i][0] = 0;
	}
	m_chMeshShuttle[0] = 0;
	m_pPanels[0] = new CMainPanel(this, 0);
	m_pPanels[1] = new CUpperPanel(this, 1);

	m_dOverSpeed = 0.0;
	m_vecLastHorzAirSpd = _V(0,0,0);

	m_hExhPtclTex = oapiRegisterExhaustTexture("contrail_blue");
	m_hLightGreenTex = oapiRegisterExhaustTexture("JTM_Strobe_Green");
	m_hLightRedTex = oapiRegisterExhaustTexture("JTM_Strobe_Red");
	m_bNavLight = m_bBcnLight = m_bStrbLight = false;
	m_dHullTemp = 0.0;
	m_bHandleHoverKeys = false;
}

// Destructor
ShuttleA::~ShuttleA ()
{
	for(int nIndex=0;nIndex < 8;nIndex++)
		oapiDeleteMesh(m_arrmhCargoSlots[nIndex]);
	for(nIndex=0;nIndex<2;nIndex++)
		delete m_arrpAuxThrusters[nIndex];
	for(nIndex=0;nIndex<2;nIndex++)
		delete m_pPanels[nIndex];
	for(nIndex=0;nIndex<4;nIndex++)
		delete m_arrMeshesCargoSlots[nIndex];
	delete m_pRadar;
}

// Set vessel class caps
void ShuttleA::SetClassCaps (FILEHANDLE cfg)
{
	char *line;


	SetSize (17.0);
	SetPMI (_V(86.6, 89.8, 5.5));
	SetEmptyMass (EMPTY_MASS);
	SetCW (0.2, 0.2, 1.5, 1.5);
	SetCrossSections (_V(132.2, 237.9, 42.4));
	SetRotDrag (_V(0.7, 0.7, 0.3));
	SetSurfaceFrictionCoeff (0.5, 0.5);
	SetCameraOffset (_V(0,2.3,16));
	SetDockParams (_V(0,0,18.32), _V(0,0,1), _V(0,1,0));
	SetTouchdownPoints (_V(0,-2.93,9), _V(-2,-2.93,-8), _V(2,-2.93,-8));
	EnableTransponder (true);


	while(oapiReadScenario_nextline(cfg, line))
	{
		if(0 == strnicmp(line, "SoundBase", 9))
			sscanf(line, "SoundBase=%d", &g_iSoundBase);
		else if(0 == strnicmp(line, "HandleHoverKeys", 15))
		{
			int iHandleKeys;
			sscanf(line, "HandleHoverKeys=%d", &iHandleKeys);
			m_bHandleHoverKeys = (iHandleKeys != 0);
		}
	}
	if(g_iSoundBase > 60 - NUM_SOUNDS)
		g_iSoundBase = 60 - NUM_SOUNDS; 
	
	if(ConnectToOrbiterSoundDLL())
	{
		RequestLoadVesselWave(SND_LCD_BEEP,"sound\\Shuttle-A2\\lcd_beep.wav");
		RequestLoadVesselWave(SND_OVRSPD_ALARM,"sound\\Shuttle-A2\\ovrspd_alarm.wav");
		RequestLoadVesselWave(SND_DMG_ALARM,"sound\\Shuttle-A2\\dmg_alarm.wav");
		RequestLoadVesselWave(SND_VIBRATION,"sound\\Shuttle-A2\\vibration.wav");
		RequestLoadVesselWave(SND_VIBRATION_HEAVY,"sound\\Shuttle-A2\\vibration_heavy.wav");
		RequestLoadVesselWave(SND_DMG_BREAK,"sound\\Shuttle-A2\\break.wav");
		RequestLoadVesselWave(SND_DMG_AUX,"sound\\Shuttle-A2\\auxexplosion.wav");
	}


	// **************************** NAV radios *************************************

	InitNavRadios (2);

	// ************************* propellant specs **********************************



	// *********************** thruster definitions ********************************


	// ******************************** mesh ***************************************

}

// Read status from scenario file
void ShuttleA::LoadState (FILEHANDLE scn, void *vs)
{
	char *line;
	double arrdPodAngle[2];

	while (oapiReadScenario_nextline (scn, line)) {
		if (!strnicmp (line, "PODANGLE", 8)) {
			sscanf (line+8, "%lf%lf", arrdPodAngle+0, arrdPodAngle+1);
			for(int nIndex = 0;nIndex < 2;nIndex++)
				m_arrpAuxThrusters[nIndex]->ForceThrustDir(arrdPodAngle[nIndex]);
		} else if (!strnicmp (line, "DOCKSTATE", 9)) {
			sscanf (line+9, "%d%lf", &dock_status, &dock_proc);
		} else if (!strnicmp (line, "AIRLOCK", 7)) {
			sscanf (line+7, "%d%lf", &lock_status, &lock_proc);
		} else if (!strnicmp (line, "CARGOCONF", 9)) {
			sscanf (line+9, "%d %d %d %d", &m_arriCargoConf[0],
				&m_arriCargoConf[1], &m_arriCargoConf[2], &m_arriCargoConf[3]);
		} else if (!strnicmp (line, "EXT_PRPLEVEL", 12)) {
			double arrdTankLvls[6];
			sscanf (line+12, "%lf %lf %lf %lf %lf %lf", &arrdTankLvls[0],
				&arrdTankLvls[1], &arrdTankLvls[2], &arrdTankLvls[3], 
				&arrdTankLvls[4], &arrdTankLvls[5]);
			for(int iIndex = 0;iIndex < 6;iIndex++)
				m_FuelManage.SetTankLvlNormalized(iIndex, arrdTankLvls[iIndex]);
		} else if (!strnicmp (line, "VALVES", 6)) {
			m_FuelManage.LoadValveStates(line + 7);
		} else if (!strnicmp (line, "DAMAGE", 6)) {
			int iAuxPodDmg, iRadarWrecked, iShuttleBroken;
			sscanf (line + 6, "%d %d %d", &iAuxPodDmg, &iRadarWrecked, &iShuttleBroken);
			m_bAuxPodDmg = iAuxPodDmg?true:false;
			m_pRadar->SetWrecked(iRadarWrecked?true:false);
			m_bShuttleBroken = (iShuttleBroken == 1)?true:false;
		} else if (!strnicmp (line, "RADAR", 5)) {
			int iRadarActive = 0;
			int iRadarScaleIndex = 0;
			double dRadarAngle = 0;
			sscanf (line + 5, "%d %lf %d", &iRadarActive, &dRadarAngle, &iRadarScaleIndex);
			m_pRadar->SetActive(iRadarActive?true:false);
			m_pRadar->SetCurAngle(dRadarAngle);
			((CUpperPanel*)m_pPanels[1])->SetRadarScaleIndex(iRadarScaleIndex);
		} else if (!strnicmp (line, "LIGHTS", 6)) {
			int iNavLight = 0;
			int iBcnLight = 0;
			int iStrbLight = 0;
			sscanf (line + 6, "%d %d %d", &iNavLight, &iBcnLight, &iStrbLight);
			m_bNavLight = (iNavLight != 0);
			m_bBcnLight = (iBcnLight != 0);
			m_bStrbLight = (iStrbLight != 0);
		} else if (!strnicmp (line, "CARGOWEIGHTS", 12)) {
			sscanf (line + 12, "%lf %lf %lf %lf", &m_arrdCargoWeights[0], 
				&m_arrdCargoWeights[1], &m_arrdCargoWeights[2], 
				&m_arrdCargoWeights[3]);
			for(int i=0;i<4;i++)
			{
				if(m_arrdCargoWeights[i] > 20000.0)
					m_arrdCargoWeights[i] = 20000.0; // 15 tons max
			}
		} else if (!strnicmp (line, "MESHNAMES", 9)) {
			sscanf (line + 9, "%s %s %s %s %s", m_chMeshShuttle, 
				m_arrMeshesCargoSlots[0], m_arrMeshesCargoSlots[1], 
				m_arrMeshesCargoSlots[2], m_arrMeshesCargoSlots[3]);
		}
		else {
			ParseScenarioLineEx (line, vs);
			// unrecognised option - pass to Orbiter's generic parser
		}
	}
}

// Write status to scenario file
void ShuttleA::SaveState (FILEHANDLE scn)
{
	char cbuf[256];

	// default vessel parameters
	SaveDefaultState (scn);

	// custom parameters
	sprintf (cbuf, "%0.4f %0.4f", m_arrpAuxThrusters[0]->GetThrustDirWant(), m_arrpAuxThrusters[0]->GetThrustDirWant());
	oapiWriteScenario_string (scn, "PODANGLE", cbuf);

	sprintf (cbuf, "%d %0.4f", dock_status, dock_proc);
	oapiWriteScenario_string (scn, "DOCKSTATE", cbuf);

	sprintf (cbuf, "%d %0.4f", lock_status, lock_proc);
	oapiWriteScenario_string (scn, "AIRLOCK", cbuf);

	sprintf (cbuf, "%d %d %d %d", m_arriCargoConf[0], m_arriCargoConf[1], 
		m_arriCargoConf[2], m_arriCargoConf[3]);
	oapiWriteScenario_string (scn, "CARGOCONF", cbuf);

	sprintf (cbuf, "%0.1lf %0.1lf %0.1lf %0.1lf", 
		m_arriCargoConf[0] == CARGO?m_arrdCargoWeights[0]:0.0, 
		m_arriCargoConf[1] == CARGO?m_arrdCargoWeights[1]:0.0, 
		m_arriCargoConf[2] == CARGO?m_arrdCargoWeights[2]:0.0, 
		m_arriCargoConf[3] == CARGO?m_arrdCargoWeights[3]:0.0);
	oapiWriteScenario_string (scn, "CARGOWEIGHTS", cbuf);

	if(strlen(m_chMeshShuttle) != 0)
	{
		sprintf (cbuf, "%s %s %s %s %s", m_chMeshShuttle, 
			m_arrMeshesCargoSlots[0], m_arrMeshesCargoSlots[1], 
			m_arrMeshesCargoSlots[2], m_arrMeshesCargoSlots[3]);
		oapiWriteScenario_string (scn, "MESHNAMES", cbuf);
	}

	sprintf (cbuf, "%0.6f %0.6f %0.6f %0.6f %0.6f %0.6f", 
		m_FuelManage.GetTankLvlNormalized(CFuelManagement::EXT1), 
		m_FuelManage.GetTankLvlNormalized(CFuelManagement::EXT2), 
		m_FuelManage.GetTankLvlNormalized(CFuelManagement::EXT3), 
		m_FuelManage.GetTankLvlNormalized(CFuelManagement::EXT4), 
		m_FuelManage.GetTankLvlNormalized(CFuelManagement::INT), 
		m_FuelManage.GetTankLvlNormalized(CFuelManagement::RCS));
	oapiWriteScenario_string (scn, "EXT_PRPLEVEL", cbuf);

	m_FuelManage.SaveValveStates(cbuf);
	oapiWriteScenario_string (scn, "VALVES", cbuf);

	sprintf (cbuf, "%d %d %d", m_bAuxPodDmg, m_pRadar->IsWrecked(), m_bShuttleBroken);
	oapiWriteScenario_string (scn, "DAMAGE", cbuf);

	sprintf (cbuf, "%d %f %d", m_pRadar->IsActive(), m_pRadar->GetCurAngle(), ((CUpperPanel*)m_pPanels[1])->m_iRadarScaleIndex);
	oapiWriteScenario_string (scn, "RADAR", cbuf);

	sprintf (cbuf, "%d %d %d", m_bNavLight, m_bBcnLight, m_bStrbLight);
	oapiWriteScenario_string (scn, "LIGHTS", cbuf);

}

void ShuttleA::PostCreation ()
{
	InitCargoSlots();
	m_FuelManage.Init(m_arriCargoConf, this);
	CreateThrusters();
	m_FuelManage.UpdateTankConnections();
	AddMeshes();
	DefineAnimations();

	SetAnimState (anim_dock, dock_proc);
	SetAnimState (anim_lock, lock_proc);

	if(m_bAuxPodDmg)
		AuxPodDamage();
	for (UINT i = 0; i < 2; i++) 
		m_arrpAuxThrusters[i]->PostCreation();
	AssignAuxPodsGroup(AuxPodsState());

	CreateLights();

	if(m_bShuttleBroken)
		BreakUp(true);
}

// Define animation sequences for moving parts
void ShuttleA::DefineAnimations ()
{
	UINT LeftPodGrp[4] = {MSHGRP_AuxThrusterHousingLeft,MSHGRP_AuxThrusterLeft,MSHGRP_AuxThrusterLitLeft,MSHGRP_AuxThrusterGridLeft};
	UINT RightPodGrp[4] = {MSHGRP_AuxThrusterHousingRight,MSHGRP_AuxThrusterRight,MSHGRP_AuxThrusterLitRight,MSHGRP_AuxThrusterGridRight};
	m_arrpAuxThrusters[0]->RegisterAnimation(LeftPodGrp, 4, m_iLeftAuxPodMsh);
	m_arrpAuxThrusters[1]->RegisterAnimation(RightPodGrp, 4);

	UINT RadarGrp[3] = {MSHGRP_RadarAxisPiece, MSHGRP_Radar_Hold, MSHGRP_Radar_Pod};
	m_pRadar->RegisterAnimation(RadarGrp, 3, m_iRadarMsh);

	static UINT UpperDockHatch = MSHGRP_UpperDockingHatch;
	static ANIMCOMP upperhatch = {
		&UpperDockHatch, 1, 0, 1,
		0, 0.554, 18.687,
		-1, 0, 0,
		(float)PI,
		0, 0,
		MESHGROUP_TRANSFORM::ROTATE
	};
	static UINT LowerDockHatch = MSHGRP_LowerDockingHatch;
	static ANIMCOMP lowerhatch = {
		&LowerDockHatch, 1, 0, 1,
		0, -0.554, 18.687,
		1, 0, 0,
		(float)PI,
		0, 0,
		MESHGROUP_TRANSFORM::ROTATE
	};
	static UINT OuterAirlock = MSHGRP_OuterAirlock;
	static ANIMCOMP outerairlock = {
		&OuterAirlock, 1, 0, 1,
		0, 0.495, 18.622,
		1, 0, 0,
		(float)(0.4*PI),
		0, 0,
		MESHGROUP_TRANSFORM::ROTATE
	};

	anim_dock = RegisterAnimSequence (0.0);
	AddAnimComp (anim_dock, &upperhatch);
	AddAnimComp (anim_dock, &lowerhatch);

	anim_lock = RegisterAnimSequence (0.0);
	AddAnimComp (anim_lock, &outerairlock);
}

// Frame update
void ShuttleA::Timestep (double simt)
{
	// animate docking hatch
	if (dock_status >= DOOR_CLOSING) {
		double da = oapiGetSimStep() * DOCK_OPERATING_SPEED;
		if (dock_status == DOOR_CLOSING) {
			if (dock_proc > 0.0)
				dock_proc = max (0.0, dock_proc-da);
			else {
				dock_status = DOOR_CLOSED;
				m_pPanels[1]->TriggerRedrawArea (CUpperPanel::AID_DOCKINDICATOR);
			}
		} else { // hatch opening
			if (dock_proc < 1.0)
				dock_proc = min (1.0, dock_proc+da);
			else {
				dock_status = DOOR_OPEN;
				m_pPanels[1]->TriggerRedrawArea (CUpperPanel::AID_DOCKINDICATOR);
			}
		}
		SetAnimState (anim_dock, dock_proc);
	}

	// animate airlock
	if (lock_status >= DOOR_CLOSING) {
		double da = oapiGetSimStep() * AIRLOCK_OPERATING_SPEED;
		if (lock_status == DOOR_CLOSING) {
			if (lock_proc > 0.0)
				lock_proc = max (0.0, lock_proc-da);
			else {
				lock_status = DOOR_CLOSED;
				m_pPanels[1]->TriggerRedrawArea (CUpperPanel::AID_AIRLOCK1INDICATOR);
			}
		} else { // door opening
			if (lock_proc < 1.0)
				lock_proc = min (1.0, lock_proc+da);
			else {
				lock_status = DOOR_OPEN;
				m_pPanels[1]->TriggerRedrawArea (CUpperPanel::AID_AIRLOCK1INDICATOR);
			}
		}
		SetAnimState (anim_lock, lock_proc);
	}

	bool bChanged = false;
	for(int nIndex=0;nIndex<2;nIndex++)
		bChanged |= m_arrpAuxThrusters[nIndex]->Timestep(oapiGetSimStep());

	if(bChanged)
		AssignAuxPodsGroup(AuxPodsState());

	m_FuelManage.Timestep(simt);
	m_pRadar->Timestep(simt);


	if(GroundContact() && m_vecLastHorzAirSpd.y < (-15.0 * 25000.0/GetMass()) && !m_bShuttleBroken)
		BreakUp();
	GetHorizonAirspeedVector(m_vecLastHorzAirSpd);

	VECTOR3 vecAirSpd;
	double dAirDens = GetAtmDensity();
	GetShipAirspeedVector(vecAirSpd);
	double dAirSpd = sqrt(vecAirSpd.x*vecAirSpd.x+vecAirSpd.y*vecAirSpd.y+vecAirSpd.z*vecAirSpd.z);
	
	double dNewOverSpeed = (0.5 * dAirDens * dAirSpd * dAirSpd) / 8000;
	if(m_dOverSpeed < 1.0 && dNewOverSpeed >= 1.0)
		SetOvrSpeedAlarm(true);
	if(dNewOverSpeed < 1.0)
		SetOvrSpeedAlarm(false);
	m_dOverSpeed = dNewOverSpeed;

	double dNewHullTemp = (pow(dAirSpd, 3)*dAirDens/3) / 22000 + (GetAtmTemperature() - 273.15);
	if(m_dHullTemp < 700.0 && dNewHullTemp >= 700.0)
		SetHullTempAlarm(true);
	if(dNewHullTemp < 700.0)
		SetHullTempAlarm(false);
	m_dHullTemp = dNewHullTemp;

	if(oapiGetFocusObject() == GetHandle())
	{
		if((m_bOvrSpdAlarm || m_bHullTempAlarm) && GetBlinkStatus(3.0, 0.1) 
			&& !IsPlaying(SND_OVRSPD_ALARM))
			PlayVesselWave(SND_OVRSPD_ALARM);
		
		if((m_bDmgAlarm || m_bRadDmgAlarm) && GetBlinkStatus(2.0, 0.1) && !IsPlaying(SND_DMG_ALARM))
			PlayVesselWave(SND_DMG_ALARM);

		if(m_dOverSpeed > 0.5 && !m_bShuttleBroken)
		{
			PlayVesselWave(SND_VIBRATION, LOOP, (int) min(255.0, (m_dOverSpeed - 0.5)*510) , (m_dOverSpeed - 0.5) * 4000 + 22050);
		}	
		else 
		{
			if(IsPlaying(SND_VIBRATION))
				StopVesselWave(SND_VIBRATION);
		}
	}

	if(m_bAuxPodDmg)
		SetThrusterLevel(th_pod[0], 1.0);

	if(!m_pRadar->IsWrecked() && m_dOverSpeed >= 1.1)
	{
		double dRadarWrkProb = pow((max(0.0, m_dOverSpeed-1.1)*2), 2);
		if(rand() < (int) (dRadarWrkProb * oapiGetSimStep() * (double) RAND_MAX))
			WreckRadar();
	}

	if(!m_bAuxPodDmg && /*(m_dOverSpeed >= 1.2 || */ m_dHullTemp >= 750) // Thruster Damage on Overspeed removed (unrealistic)
	{
		double dAuxPodDmgProb = /*max(pow((max(0.0, m_dOverSpeed-1.2)), 2),*/ pow((max(0.0, (m_dHullTemp-750.0)/100.0)), 2); //);
		if(rand() < (int) (dAuxPodDmgProb * oapiGetSimStep() * (double) RAND_MAX))
			AuxPodDamage();
	}

	if(!m_bShuttleBroken && (m_dOverSpeed >= 1.5 || m_dHullTemp >= 800))
	{
		double dBreakProb = max(pow((max(0.0, m_dOverSpeed-1.5)), 2), pow(max(0.0, (m_dHullTemp - 800.0) / 200.0), 2));
		if(rand() < (int) (dBreakProb * oapiGetSimStep() * (double) RAND_MAX))
			BreakUp();
	}

	bool bLightsOff = (GroundContact()||m_bShuttleBroken);
	bool bBlinkStatus = GetBlinkStatus(1.0, 0.1);
	SetThrusterLevel(m_thLights[LGT_RIGHT_STRB], (bBlinkStatus&&!bLightsOff&&m_bStrbLight)?1.0:0.0);
	SetThrusterLevel(m_thLights[LGT_LEFT_STRB], (bBlinkStatus&&!bLightsOff&&m_bStrbLight)?1.0:0.0);
	SetThrusterLevel(m_thLights[LGT_AFT_STRB], (bBlinkStatus&&!bLightsOff&&m_bStrbLight)?1.0:0.0);
	SetThrusterLevel(m_thLights[LGT_TOP_BCN], (!bLightsOff&&GetBlinkStatus(0.5, 0.05, 0.1)&&m_bBcnLight)?1.0:0.0);
	SetThrusterLevel(m_thLights[LGT_BTM_BCN], (!bLightsOff&&GetBlinkStatus(0.5, 0.05, 0.15)&&m_bBcnLight)?1.0:0.0);
	SetThrusterLevel(m_thLights[LGT_RIGHT], (!bLightsOff&&m_bNavLight)?1.0:0.0);
	SetThrusterLevel(m_thLights[LGT_LEFT], (!bLightsOff&&m_bNavLight)?1.0:0.0);
	SetThrusterLevel(m_thLights[LGT_AFT], (!bLightsOff&&m_bNavLight)?1.0:0.0);

}

void ShuttleA::RotatePod (UINT which, UINT mode)
{
	switch(mode)
	{
	case 0:
		m_arrpAuxThrusters[which]->RotateStop();
		break;
	case 1:
		m_arrpAuxThrusters[which]->RotateBackward();
		break;
	case 2:
		m_arrpAuxThrusters[which]->RotateForward();
		break;
	}
}

void ShuttleA::ActivateDockingPort (DoorStatus action)
{
	dock_status = action;
	m_pPanels[1]->TriggerRedrawArea (CUpperPanel::AID_DOCKSWITCH);
	m_pPanels[1]->TriggerRedrawArea (CUpperPanel::AID_DOCKINDICATOR);
}

void ShuttleA::RevertDockingPort ()
{
	ActivateDockingPort (dock_status == DOOR_CLOSED || dock_status == DOOR_CLOSING ?
						 DOOR_OPENING : DOOR_CLOSING);
}

void ShuttleA::ActivateAirlock (DoorStatus action)
{
	lock_status = action;
	m_pPanels[1]->TriggerRedrawArea (CUpperPanel::AID_AIRLOCK1SWITCH);
	m_pPanels[1]->TriggerRedrawArea (CUpperPanel::AID_AIRLOCK1INDICATOR);
}

void ShuttleA::RevertAirlock ()
{
	ActivateAirlock (lock_status == DOOR_CLOSED || lock_status == DOOR_CLOSING ?
		             DOOR_OPENING : DOOR_CLOSING);
}

UINT ShuttleA::AuxPodsState()
{
	double arrdAngle[2], arrdAngleWant[2];
	for(int nIndex = 0;nIndex < 2;nIndex++)
	{
		arrdAngle[nIndex] = m_arrpAuxThrusters[nIndex]->GetThrustDir();
		arrdAngleWant[nIndex] = m_arrpAuxThrusters[nIndex]->GetThrustDirWant();
	}

	if(arrdAngle[0] == arrdAngle[1] && arrdAngle[0] == arrdAngleWant[0])
	{
		if(arrdAngle[0] >= 0.0 && arrdAngle[0] <= 0.0001) // Retro Position 
			return RETRO;
		else if(arrdAngle[0] >= PI05-0.0001 && arrdAngle[0] <= PI05+0.0001) // Hover Pos
			return HOVER;
		else if(arrdAngle[0] >= PI-0.0001 && arrdAngle[0] <= PI) // Main/Forward Pos
			return MAIN;
	}

	if(arrdAngleWant[0] == arrdAngleWant[1])
	{
		if(arrdAngleWant[0] == 0.0)
			return MOVETO_RETRO;
		else if(arrdAngleWant[0] == PI05) // Hover Pos
			return MOVETO_HOVER;
		else if(arrdAngleWant[0] == PI) // Main/Forward Pos
			return MOVETO_MAIN;
	}

	return USER;
}

void ShuttleA::AssignAuxPodsGroup(UINT uiNewGroup)
{
	if(m_uiCurPodsGroup == uiNewGroup)
		return;
	switch(m_uiCurPodsGroup)
	{
	case USER:
		DelThrusterGroup(thg_pod, THGROUP_USER);
		break;
	case RETRO:
		DelThrusterGroup(thg_retro, THGROUP_RETRO);
		thg_retro = CreateThrusterGroup(th_pod, 0, THGROUP_RETRO);
		break;
	case HOVER:
		DelThrusterGroup(thg_hover, THGROUP_HOVER);
		thg_hover = CreateThrusterGroup(th_hover, 2, THGROUP_HOVER);
		break;
	case MAIN:
		DelThrusterGroup(thg_main, THGROUP_MAIN);
		thg_main = CreateThrusterGroup(th_main, 2, THGROUP_MAIN);
		break;
	}

	switch(uiNewGroup)
	{
	case USER:
	case MOVETO_RETRO:
	case MOVETO_HOVER:
	case MOVETO_MAIN:
		thg_pod = CreateThrusterGroup(th_pod, 2, THGROUP_USER);
		m_uiCurPodsGroup = USER;
		break;
	case RETRO:
		thg_retro = CreateThrusterGroup(th_pod, 2, THGROUP_RETRO);
		m_uiCurPodsGroup = RETRO;
		break;
	case HOVER:
		DelThrusterGroup(thg_hover, THGROUP_HOVER);
		thg_hover = CreateThrusterGroup(th_hover, 4, THGROUP_HOVER);
		m_uiCurPodsGroup = HOVER;
		break;
	case MAIN:
		DelThrusterGroup(thg_main, THGROUP_MAIN);
		thg_main = CreateThrusterGroup(th_main, 4, THGROUP_MAIN);
		m_uiCurPodsGroup = MAIN;
		break;
	}
}

void ShuttleA::ToggleAuxThrustMode(int iButton)
{
	double dNewAngle;
	switch(iButton)
	{
	case 0:
		dNewAngle = 0.0;
		break;
	case 1: 
		dNewAngle = PI05;
		break;
	case 2:
		dNewAngle = PI;
		break;
	}

	for(int nIndex = 0;nIndex < 2; nIndex++)
		m_arrpAuxThrusters[nIndex]->SetThrustDirWant(dNewAngle);
}

void ShuttleA::InitCargoSlots()
{
	// Add Meshes of Tanks or Cargo Containers as specified in the config file
	// and calculate mass, fuel status and crossections accordingly
	double dCrsSecZ, arrdCrsSecX[4], dCrsSecY;
	VECTOR3 vecCrsSec, vecAllCrsSec;
	int nIndex;

	for(nIndex = 0;nIndex < 4;nIndex++)
		arrdCrsSecX[nIndex] = 0.0;
	dCrsSecY = 0.0;
	dCrsSecZ = 0.0;
	double dMass = EMPTY_MASS;
	for(nIndex = 0;nIndex<4;nIndex++)
	{
		switch(m_arriCargoConf[nIndex])
		{
		case EMPTY:
			m_arrmhCargoSlots[nIndex*2] = oapiLoadMesh(BuildFilename("Left", nIndex));
			m_arrmhCargoSlots[nIndex*2+1] = oapiLoadMesh(BuildFilename("Right", nIndex));
			dMass += STABILIZER_MASS;
			vecCrsSec = CRSSEC_STABILIZER;
			break;
		case TANK:
			m_arrmhCargoSlots[nIndex*2] = oapiLoadMesh(BuildFilename("Left", nIndex));
			m_arrmhCargoSlots[nIndex*2+1] = oapiLoadMesh(BuildFilename("Right", nIndex));
			dMass += TANK_MASS;
			m_iNumExtTanks++;
			vecCrsSec = CRSSEC_TANK;
			break;
		case CARGO:
			m_arrmhCargoSlots[nIndex*2] = oapiLoadMesh(BuildFilename("Left", nIndex));
			m_arrmhCargoSlots[nIndex*2+1] = oapiLoadMesh(BuildFilename("Right", nIndex));
			dMass += (CARGO_MASS + m_arrdCargoWeights[nIndex]);
			vecCrsSec = CRSSEC_CARGO;
			break;
		}
		dCrsSecZ = max(dCrsSecZ, vecCrsSec.z);
		arrdCrsSecX[nIndex] = max(arrdCrsSecX[nIndex], vecCrsSec.x);
		dCrsSecY += vecCrsSec.y;
	}

	SetEmptyMass (dMass);
	vecAllCrsSec = CRSSEC_EMPTY_SHUTTLE;
	vecAllCrsSec.z += dCrsSecZ;
	for(nIndex = 0;nIndex < 4;nIndex++)
		vecAllCrsSec.x += arrdCrsSecX[nIndex];
	vecAllCrsSec.y += dCrsSecY;
	SetCrossSections (vecAllCrsSec);
}

bool ShuttleA::LoadPanel(int iID)
{
	bool bRet = m_pPanels[iID]->Load();
	m_pPanels[iID]->Init();

	return bRet;
}

bool ShuttleA::RedrawPanel(int iID, int iEvent, SURFHANDLE sh)
{
	bool bRet = false;
	for(int iIndex = 0;iIndex < 2;iIndex++)
		bRet |= m_pPanels[iIndex]->Redraw(iID, iEvent, sh);

	return bRet;
}

bool ShuttleA::PanelMouseEvent(int iID, int iEvent, int iX, int iY)
{
	bool bRet = false;
	for(int iIndex = 0;iIndex < 2;iIndex++)
		bRet |= m_pPanels[iIndex]->MouseEvent(iID, iEvent, iX, iY);

	return bRet;
}

void ShuttleA::MFDmode(int iMfd, int iMode)
{
	switch (iMfd) 
	{
	case MFD_LEFT:
		m_pPanels[0]->TriggerRedrawArea (CMainPanel::AID_MFD1_LBUTTONS);
		m_pPanels[0]->TriggerRedrawArea (CMainPanel::AID_MFD1_RBUTTONS);
		break;
	case MFD_RIGHT:
		m_pPanels[0]->TriggerRedrawArea (CMainPanel::AID_MFD2_LBUTTONS);
		m_pPanels[0]->TriggerRedrawArea (CMainPanel::AID_MFD2_RBUTTONS);
		break;
	}

}

void ShuttleA::Navmode(int iMode, bool bActive)
{
	m_pPanels[0]->TriggerRedrawArea (CMainPanel::AID_NAVMODE);
}

int ShuttleA::ConsumeBufferedKey(DWORD dwKey, bool bDown, char *kstate)
{
	int nIndex;

	if (bDown)
	{
		if (KEYMOD_SHIFT (kstate)) {
		} else if (KEYMOD_CONTROL (kstate)) {
			switch (dwKey) {
			case OAPI_KEY_DIVIDE:
				if (SetAttitudeMode (GetAttitudeMode() >= 1 ? 0 : 1))
					m_pPanels[0]->TriggerRedrawArea (CMainPanel::AID_ATTITUDEMODE);
				return 1;
			}
		} 
		else {
			switch (dwKey) {
			case OAPI_KEY_NUMPAD0:
				if(m_bHandleHoverKeys)
				{
					m_dOldHoverLvl = GetThrusterGroupLevel(THGROUP_HOVER);
					SetThrusterGroupLevel(THGROUP_HOVER, 1.0);
					return 1;
				}
			case OAPI_KEY_DECIMAL:
				if(m_bHandleHoverKeys)
				{
					SetThrusterGroupLevel(THGROUP_HOVER, 0.0);
					return 1;
				}
			case OAPI_KEY_K:  // "operate docking port"
				RevertDockingPort ();
				return 1;
			case OAPI_KEY_O:  // "operate outer airlock"
				RevertAirlock ();
				return 1;
			case OAPI_KEY_DIVIDE:
				if (ToggleAttitudeMode()) 
					m_pPanels[0]->TriggerRedrawArea (CMainPanel::AID_ATTITUDEMODE);
				return 1;
			case OAPI_KEY_1:
				for(nIndex = 0;nIndex < 2; nIndex++)
					m_arrpAuxThrusters[nIndex]->SetThrustDirWant(PI);
				return 1;
			case OAPI_KEY_2:
				for(nIndex = 0;nIndex < 2; nIndex++)
					m_arrpAuxThrusters[nIndex]->SetThrustDirWant(PI/2);
				return 1;
			case OAPI_KEY_3:
				for(nIndex = 0;nIndex < 2; nIndex++)
					m_arrpAuxThrusters[nIndex]->SetThrustDirWant(0.0);
				return 1;
			case OAPI_KEY_4:
				RotatePod (0, 2);
				RotatePod (1, 2);
				return 1;
			case OAPI_KEY_5:
				RotatePod (0, 1);
				RotatePod (1, 1);
				return 1;
	#ifdef _DEBUG
			case OAPI_KEY_D:
				AuxPodDamage();
				return 1;
	//		case OAPI_KEY_R:
	//			WreckRadar();
	//			return 1;
			case OAPI_KEY_B:
				BreakUp();
				return 1; 
			case OAPI_KEY_S:
				AddParticleStreams();
				return 1;
	#endif
			}
		}
	}
	else
	{
		switch (dwKey) 
		{
		case OAPI_KEY_4:
		case OAPI_KEY_5:
			RotatePod(0, 0);
			RotatePod(1, 0);
			return 1;
		case OAPI_KEY_NUMPAD0:
			if(m_bHandleHoverKeys)
			{
				SetThrusterGroupLevel(THGROUP_HOVER, m_dOldHoverLvl);
				return 1;
			}
		}
	}

	return 0;
}

void ShuttleA::CreateThrusters()
{
	SetISP (ISP);
	// main thrusters
	th_main[0] = CreateThruster (_V(-1.8,0,-16.15), _V(0,0,1), MAX_MAIN_THRUST, m_FuelManage.m_arrTanks[CFuelManagement::INT].m_ph, ISP);
	th_main[1] = CreateThruster (_V( 1.8,0,-16.15), _V(0,0,1), MAX_MAIN_THRUST, m_FuelManage.m_arrTanks[CFuelManagement::INT].m_ph, ISP);
	thg_main = CreateThrusterGroup (th_main, 2, THGROUP_MAIN);
	AddExhaust (th_main[0], 12, 1);
	AddExhaust (th_main[1], 12, 1);

	// hover thrusters
	th_hover[0] = CreateThruster (_V(0,-2.4, 12.83), _V(0,1,0), MAX_HOVER_THRUST, m_FuelManage.m_arrTanks[CFuelManagement::INT].m_ph, ISP);
	th_hover[1] = CreateThruster (_V(0,-2.4,-12.83), _V(0,1,0), MAX_HOVER_THRUST, m_FuelManage.m_arrTanks[CFuelManagement::INT].m_ph, ISP);
	thg_hover = CreateThrusterGroup (th_hover, 2, THGROUP_HOVER);
	AddExhaust (th_hover[0], 10, 1);
	AddExhaust (th_hover[1], 10, 1);

	// rotateable aux thrusters 
	for(int nIndex = 0;nIndex < 2;nIndex++)
		th_main[2+nIndex] = th_hover[2+nIndex] = th_pod[nIndex] = m_arrpAuxThrusters[nIndex]->Create(m_FuelManage.m_arrTanks[CFuelManagement::INT].m_ph);

	thg_pod = CreateThrusterGroup (th_pod, 2, THGROUP_USER);
	

	// attitude thrusters
	THRUSTER_HANDLE th_att_rot[4], th_att_lin[4];
	m_arrthAtt[0] = th_att_rot[0] = CreateThruster (_V(-6, 0.7,0.0), _V(0,-1,0), MAX_RCS_THRUST, m_FuelManage.m_arrTanks[CFuelManagement::RCS].m_ph, ISP);
	m_arrthAtt[1] = th_att_rot[1] = CreateThruster (_V( 6,-0.7,0.0), _V(0, 1,0), MAX_RCS_THRUST, m_FuelManage.m_arrTanks[CFuelManagement::RCS].m_ph, ISP);
	m_arrthAtt[2] = th_att_rot[2] = CreateThruster (_V(-6,-0.7,0.0), _V(0, 1,0), MAX_RCS_THRUST, m_FuelManage.m_arrTanks[CFuelManagement::RCS].m_ph, ISP);
	m_arrthAtt[3] = th_att_rot[3] = CreateThruster (_V( 6, 0.7,0.0), _V(0,-1,0), MAX_RCS_THRUST, m_FuelManage.m_arrTanks[CFuelManagement::RCS].m_ph, ISP);
	CreateThrusterGroup (th_att_rot,   2, THGROUP_ATT_BANKLEFT);
	CreateThrusterGroup (th_att_rot+2, 2, THGROUP_ATT_BANKRIGHT);
	AddExhaust (th_att_rot[0], 0.7, 0.08, _V(-6, 0.7, 0.15), _V(0,1,0));
	AddExhaust (th_att_rot[0], 0.7, 0.08, _V(-6, 0.7, -0.15), _V(0,1,0));
	AddExhaust (th_att_rot[1], 0.7, 0.08, _V( 6, -0.7, 0.15), _V(0, -1,0));
	AddExhaust (th_att_rot[1], 0.7, 0.08, _V( 6, -0.7, -0.15), _V(0, -1,0));
	AddExhaust (th_att_rot[2], 0.7, 0.08, _V(-6, -0.7, 0.15), _V(0, -1,0));
	AddExhaust (th_att_rot[2], 0.7, 0.08, _V(-6, -0.7, -0.15), _V(0, -1,0));
	AddExhaust (th_att_rot[3], 0.7, 0.08, _V( 6, 0.7, 0.15), _V(0, 1,0));
	AddExhaust (th_att_rot[3], 0.7, 0.08, _V( 6, 0.7, -0.15), _V(0, 1,0));
	m_arrthAtt[4] = th_att_rot[0] = th_att_lin[0] = CreateThruster (_V(0,0, 15), _V(0, 1,0), MAX_RCS_THRUST, m_FuelManage.m_arrTanks[CFuelManagement::RCS].m_ph, ISP);
	m_arrthAtt[5] = th_att_rot[1] = th_att_lin[2] = CreateThruster (_V(0,0,-15), _V(0,-1,0), MAX_RCS_THRUST, m_FuelManage.m_arrTanks[CFuelManagement::RCS].m_ph, ISP);
	m_arrthAtt[6] = th_att_rot[2] = th_att_lin[3] = CreateThruster (_V(0,0, 15), _V(0,-1,0), MAX_RCS_THRUST, m_FuelManage.m_arrTanks[CFuelManagement::RCS].m_ph, ISP);
	m_arrthAtt[7] = th_att_rot[3] = th_att_lin[1] = CreateThruster (_V(0,0,-15), _V(0, 1,0), MAX_RCS_THRUST, m_FuelManage.m_arrTanks[CFuelManagement::RCS].m_ph, ISP);
	CreateThrusterGroup (th_att_rot,   2, THGROUP_ATT_PITCHUP);
	CreateThrusterGroup (th_att_rot+2, 2, THGROUP_ATT_PITCHDOWN);
	CreateThrusterGroup (th_att_lin,   2, THGROUP_ATT_UP);
	CreateThrusterGroup (th_att_lin+2, 2, THGROUP_ATT_DOWN);
	AddExhaust (th_att_rot[0], 0.7, 0.08, _V(-0.2, -1.1,   17.9), _V(0,-1,0));
	AddExhaust (th_att_rot[0], 0.7, 0.08, _V( 0.2, -1.1,   17.9), _V(0,-1,0));
	AddExhaust (th_att_rot[1], 0.7, 0.08, _V( 0,    2.3,-14.2), _V(0, 1,0));
	AddExhaust (th_att_rot[1], 0.7, 0.08, _V( 0,    2.3,-13.9), _V(0, 1,0));
	AddExhaust (th_att_rot[2], 0.7, 0.08, _V(-0.15, 2.1, 18.0), _V(0, 1,0));
	AddExhaust (th_att_rot[2], 0.7, 0.08, _V( 0.15, 2.1, 18.0), _V(0, 1,0));
	AddExhaust (th_att_rot[3], 0.7, 0.08, _V(-0.15,-1.9,-14.1), _V(0,-1,0));
	AddExhaust (th_att_rot[3], 0.7, 0.08, _V( 0.15,-1.9,-14.1), _V(0,-1,0));
	m_arrthAtt[8] = th_att_rot[0] = th_att_lin[0] = CreateThruster (_V(0,0, 15), _V( 1,0,0), MAX_RCS_THRUST, m_FuelManage.m_arrTanks[CFuelManagement::RCS].m_ph, ISP);
	m_arrthAtt[9] = th_att_rot[1] = th_att_lin[2] = CreateThruster (_V(0,0,-15), _V(-1,0,0), MAX_RCS_THRUST, m_FuelManage.m_arrTanks[CFuelManagement::RCS].m_ph, ISP);
	m_arrthAtt[10] = th_att_rot[2] = th_att_lin[3] = CreateThruster (_V(0,0, 15), _V(-1,0,0), MAX_RCS_THRUST, m_FuelManage.m_arrTanks[CFuelManagement::RCS].m_ph, ISP);
	m_arrthAtt[11] = th_att_rot[3] = th_att_lin[1] = CreateThruster (_V(0,0,-15), _V( 1,0,0), MAX_RCS_THRUST, m_FuelManage.m_arrTanks[CFuelManagement::RCS].m_ph, ISP);
	CreateThrusterGroup (th_att_rot,   2, THGROUP_ATT_YAWRIGHT);
	CreateThrusterGroup (th_att_rot+2, 2, THGROUP_ATT_YAWLEFT);
	CreateThrusterGroup (th_att_lin,   2, THGROUP_ATT_RIGHT);
	CreateThrusterGroup (th_att_lin+2, 2, THGROUP_ATT_LEFT);
	AddExhaust (th_att_rot[0], 0.7, 0.08, _V(-2.4, 0.15, 17.9), _V(-1,0,0));
	AddExhaust (th_att_rot[0], 0.7, 0.08, _V(-2.4,-0.15, 17.9), _V(-1,0,0));
	AddExhaust (th_att_rot[1], 0.7, 0.08, _V( 3.5, 0,   -14.2), _V( 1,0,0));
	AddExhaust (th_att_rot[1], 0.7, 0.08, _V( 3.5, 0,   -13.9), _V( 1,0,0));
	AddExhaust (th_att_rot[2], 0.7, 0.08, _V( 2.4, 0.15, 17.9), _V( 1,0,0));
	AddExhaust (th_att_rot[2], 0.7, 0.08, _V( 2.4,-0.15, 17.9), _V( 1,0,0));
	AddExhaust (th_att_rot[3], 0.7, 0.08, _V(-3.5, 0,   -14.2), _V(-1,0,0));
	AddExhaust (th_att_rot[3], 0.7, 0.08, _V(-3.5, 0,   -13.9), _V(-1,0,0));
	m_arrthAtt[12] = th_att_lin[0] = CreateThruster (_V( 0,0,0), _V(0,0, 1), 2*MAX_RCS_THRUST, m_FuelManage.m_arrTanks[CFuelManagement::RCS].m_ph, ISP);
	m_arrthAtt[13] = th_att_lin[1] = CreateThruster (_V( 0,0,0), _V(0,0,-1), 2*MAX_RCS_THRUST, m_FuelManage.m_arrTanks[CFuelManagement::RCS].m_ph, ISP);
	CreateThrusterGroup (th_att_lin,   1, THGROUP_ATT_FORWARD);
	CreateThrusterGroup (th_att_lin+1, 1, THGROUP_ATT_BACK);
	AddExhaust (th_att_lin[0], 1.4, 0.16, _V( 6,0,-1.2), _V(0,0,-1));
	AddExhaust (th_att_lin[0], 1.4, 0.16, _V(-6,0,-1.2), _V(0,0,-1));
	AddExhaust (th_att_lin[1], 1.4, 0.16, _V( 6,0, 1.2), _V(0,0, 1));
	AddExhaust (th_att_lin[1], 1.4, 0.16, _V(-6,0, 1.2), _V(0,0, 1));

	AddParticleStreams();
}

void ShuttleA::AddMeshes(bool bReload)
{
	VECTOR3 vecSlotOffset;

	oapiLoadMeshGlobal(BuildFilename("BrkFront"));
	oapiLoadMeshGlobal(BuildFilename("BrkRear"));

	if(bReload)
		ClearMeshes();

	if(m_bShuttleBroken)
	{
		AddMesh (oapiLoadMeshGlobal (BuildFilename("BrkFront")));
	}
	else
	{
		AddMesh (oapiLoadMeshGlobal (BuildFilename("")));		
		if(m_bAuxPodDmg)
		{
			m_iLeftAuxPodMsh = AddMesh(oapiLoadMeshGlobal(BuildFilename("ThrDmg")));
			if(!m_bThrDmgAdded)
			{
				m_arrpAuxThrusters[0]->MeshChanged(m_iLeftAuxPodMsh);
				m_bThrDmgAdded = true;
			}
		}
		else
			m_iLeftAuxPodMsh = AddMesh(oapiLoadMeshGlobal(BuildFilename("Thr")));
	}

	for(int nIndex = 0;nIndex < (m_bShuttleBroken?1:4);nIndex++)
	{
		vecSlotOffset = _V(-CARGOSLOTOFFSET_X, 0.0, CARGOSLOTOFFSET_Z[nIndex]);
		AddMesh(m_arrmhCargoSlots[nIndex*2], &vecSlotOffset);
		vecSlotOffset.x = CARGOSLOTOFFSET_X;
		AddMesh(m_arrmhCargoSlots[nIndex*2+1], &vecSlotOffset);
	}

	if(m_pRadar->IsWrecked())
		m_iRadarMsh = AddMesh(oapiLoadMeshGlobal(BuildFilename("RadWrk")));	
	else
		m_iRadarMsh = AddMesh(oapiLoadMeshGlobal(BuildFilename("Rad")));
}

void ShuttleA::WreckRadar()
{
	m_pRadar->SetWrecked();
	AddMeshes(true);
	SetRadDmgAlarm(true);
}

void ShuttleA::AuxPodDamage()
{
	m_bAuxPodDmg = true;
	SetDmgAlarm(true);
	m_arrpAuxThrusters[0]->SetExhaustParams(4.5, 1.0, 1.6);
	m_arrpAuxThrusters[0]->ChangePosAndDir(_V(-6.5 ,0.0, 0.0), _V(0.3, 0.0, -0.9));
	SetThrusterMax0(th_pod[0], MAX_RETRO_THRUST * 1.24);
	m_arrpAuxThrusters[0]->SetThrustLevel(1.0);
	AddMeshes(true);
	PlayVesselWave(SND_DMG_AUX);
}

void ShuttleA::VisualCreated(VISHANDLE vh)
{
	m_vh = vh;
	for(int iIndex = 0;iIndex<2;iIndex++)
		m_arrpAuxThrusters[iIndex]->VisualCreated(vh);
}

void ShuttleA::VisualDestroyed(VISHANDLE vh)
{
	if(m_vh == vh)
	{
		m_vh = NULL;
		for(int iIndex = 0;iIndex<2;iIndex++)
			m_arrpAuxThrusters[iIndex]->VisualDestroyed(vh);
	}
}

void ShuttleA::Animate(double dSimTime)
{
	for(int iIndex=0;iIndex <2;iIndex++)
		m_arrpAuxThrusters[iIndex]->Animate(dSimTime);
}

void ShuttleA::SetOvrSpeedAlarm(bool bAlarm)
{
	m_bOvrSpdAlarm = bAlarm;
}

bool ShuttleA::GetOvrSpeedAlarm()
{
	return m_bOvrSpdAlarm;
}

void ShuttleA::SetDmgAlarm(bool bAlarm)
{
	m_bDmgAlarm = bAlarm;
}

bool ShuttleA::GetDmgAlarm()
{
	return m_bDmgAlarm;
}

void ShuttleA::BreakUp(bool bOnInit)
{
	if(!bOnInit)
	{
		if(m_arriCargoConf[1] == TANK)
		{
			CreateCargObj(m_arriCargoConf[1], true, m_FuelManage.GetTankFuelMass(CFuelManagement::EXT2));
			CreateCargObj(m_arriCargoConf[1], false, m_FuelManage.GetTankFuelMass(CFuelManagement::EXT2));
		}
		else 
		{
			CreateCargObj(m_arriCargoConf[1], true, m_arrdCargoWeights[1]);
			CreateCargObj(m_arriCargoConf[1], false, m_arrdCargoWeights[1]);
		}
		CreateRearPart();
		AddForce(_V(0,1000000,0), _V(0.5,0,20));
		PlayVesselWave(SND_DMG_BREAK);
	}
	m_bShuttleBroken = true;
	m_FuelManage.BreakUp();
	m_arrpAuxThrusters[0]->BreakUp();
	m_arrpAuxThrusters[1]->BreakUp();
	AddMeshes(true);

	SetSize (9);
	SetCW (0.2, 0.2, 1.0, 1.0);
	SetRotDrag (_V(0.5, 0.5, 0.3));
	SetTouchdownPoints (_V(2,-2.93,12.5), _V(-2,-2.93,12.5), _V(0,-1.75,6.4));
	switch(m_arriCargoConf[0])
	{
	case EMPTY:
		SetCrossSections(_V(44, 42, 22) + CRSSEC_STABILIZER);
		SetEmptyMass(FRONT_PART_MASS + STABILIZER_MASS);
		break;
	case TANK:
		SetCrossSections(_V(44, 42, 22) + CRSSEC_TANK);
		SetEmptyMass(FRONT_PART_MASS + TANK_MASS);
		break;
	case CARGO:
		SetCrossSections(_V(44, 42, 22) + CRSSEC_CARGO);
		SetEmptyMass(FRONT_PART_MASS + CARGO_MASS + m_arrdCargoWeights[0]);
		break;
	}
	SetPMI(_V(100,100,3));
}

void ShuttleA::CreateCargObj(int iType, bool bLeft, double dLoadMass)
{
	char chName[100];
	strncpy(chName, GetName(), 90);
	if(bLeft)
		strcat(chName, " left");
	else
		strcat(chName, " right");
	switch(iType)
	{
	case EMPTY:
		strcat(chName, " Stabilizer");
		break;
	case TANK:
		strcat(chName, " Tank");
		break;
	case CARGO:
		strcat(chName, " Cargo Container");
		break;
	}

	VESSELSTATUS2 vs2;
	vs2.version = 2;
	vs2.flag = 0;
	GetStatusEx(&vs2);
	VECTOR3 vecRPOSLocal, vecRVelLocal;
	Global2Local(vs2.rpos, vecRPOSLocal);
	vecRPOSLocal += _V(bLeft?-CARGOSLOTOFFSET_X:CARGOSLOTOFFSET_X, 0.0, CARGOSLOTOFFSET_Z[1]);
	Local2Global(vecRPOSLocal, vs2.rpos);

	Global2Local(vs2.rvel, vecRVelLocal);
	vecRVelLocal += _V(bLeft?-1:1.2, 0.4, 0);
	Local2Global(vecRVelLocal, vs2.rvel);

	GlobalRot(_V(bLeft?0.2:0.3, bLeft?-0.5:0.6, 0.0), vs2.vrot);

	vs2.status = 0;
	vs2.nfuel = 0;
	vs2.fuel = NULL;
	vs2.nthruster = 0;
	vs2.thruster = NULL;
	vs2.ndockinfo = 0;
	vs2.dockinfo = NULL;
	OBJHANDLE hObj = oapiCreateVesselEx (chName, "VESSEL", &vs2);
	VESSEL* pObj = oapiGetVesselInterface(hObj);

	pObj->SetSize (2.5);
	pObj->SetCW (0.8, 0.8, 0.5, 0.5);
	pObj->SetRotDrag (_V(0.5, 0.5, 0.3));
	pObj->SetSurfaceFrictionCoeff (0.5, 0.5);
	pObj->SetCameraOffset (_V(0,0,0));
	pObj->SetTouchdownPoints (_V(0.1,-1.65,-2.0), _V(-0.1,-1.65,-2.0), _V(0.0,-1.65,2.0));
	pObj->EnableTransponder (false);
	pObj->InitNavRadios (0);

	switch(iType)
	{
	case ShuttleA::EMPTY:
		pObj->AddMesh(oapiLoadMeshGlobal(BuildFilename(bLeft?"Left":"Right", 1)));
		pObj->SetPMI(_V(2.74, 2.74, 0.03));
		pObj->SetCrossSections(_V(2.2, 2.2, 0.57));
		pObj->SetEmptyMass(STABILIZER_MASS / 2);
		break;
	case ShuttleA::TANK:
		pObj->AddMesh(oapiLoadMeshGlobal(BuildFilename(bLeft?"Left":"Right", 1)));
		pObj->SetPMI(_V(2.29, 2.29, 1.27));
		pObj->SetCrossSections(_V(15.2, 15.2, 8.28));
		pObj->SetEmptyMass((TANK_MASS + dLoadMass) / 2);
		break;
	case ShuttleA::CARGO:
		pObj->AddMesh(oapiLoadMeshGlobal(BuildFilename(bLeft?"Left":"Right", 1)));
		pObj->SetPMI(_V(2.57, 2.47, 2.08));
		pObj->SetCrossSections(_V(17.5, 17.5, 13.11));
		pObj->SetEmptyMass((CARGO_MASS + dLoadMass) / 2);
		break;
	}

}

void ShuttleA::CreateRearPart()
{
	char chName[100];
	strncpy(chName, GetName(), 90);
	strcat(chName, " Rear Part");

	VESSELSTATUS2 vs2;
	vs2.version = 2;
	vs2.flag = 0;
	GetStatusEx(&vs2);

	VECTOR3 vecRVelLocal;
	Global2Local(vs2.rvel, vecRVelLocal);
	vecRVelLocal += _V(4, 0, 1);
	Local2Global(vecRVelLocal, vs2.rvel);

	GlobalRot(_V(0.4, 0, 0.0), vs2.vrot);

	vs2.status = 0;
	vs2.nfuel = 0;
	vs2.fuel = NULL;
	vs2.nthruster = 0;
	vs2.thruster = NULL;
	vs2.ndockinfo = 0;
	vs2.dockinfo = NULL;
	OBJHANDLE hObj = oapiCreateVesselEx (chName, "VESSEL", &vs2);
	VESSEL* pObj = oapiGetVesselInterface(hObj);

	pObj->SetSize (11);
	pObj->SetCW (1.5, 1.5, 1.5, 1.5);
	pObj->SetRotDrag (_V(0.8, 0.8, 0.3));
	pObj->SetSurfaceFrictionCoeff (0.5, 0.5);
	pObj->SetCameraOffset (_V(0,0,0));
	SetTouchdownPoints (_V(0,-2.93,0), _V(-2,-2.93,-8), _V(2,-2.93,-8));
	pObj->EnableTransponder (false);
	pObj->InitNavRadios (0);

	pObj->AddMesh(oapiLoadMeshGlobal(BuildFilename("BrkRear")));
	pObj->SetPMI(_V(69, 76, 10));
	VECTOR3 vecCrsSec, vecSlotOffset;

	double dMass, dCrsSecX, dCrsSecY, dCrsSecZ;

	dMass = REAR_PART_MASS;
	dCrsSecX = dCrsSecY = dCrsSecZ = 0;
	
	for(int nIndex = 2;nIndex<4;nIndex++)
	{
		switch(m_arriCargoConf[nIndex])
		{
		case EMPTY:
			vecSlotOffset = _V(-CARGOSLOTOFFSET_X, 0.0, CARGOSLOTOFFSET_Z[nIndex]);
			pObj->AddMesh(oapiLoadMeshGlobal(BuildFilename("Left", nIndex)), &vecSlotOffset);
			vecSlotOffset.x = CARGOSLOTOFFSET_X;
			pObj->AddMesh(oapiLoadMeshGlobal(BuildFilename("Right", nIndex)), &vecSlotOffset);
			dMass += STABILIZER_MASS;
			vecCrsSec = CRSSEC_STABILIZER;
			break;
		case TANK:
			vecSlotOffset = _V(-CARGOSLOTOFFSET_X, 0.0, CARGOSLOTOFFSET_Z[nIndex]);
			pObj->AddMesh(oapiLoadMeshGlobal(BuildFilename("Left", nIndex)), &vecSlotOffset);
			vecSlotOffset.x = CARGOSLOTOFFSET_X;
			pObj->AddMesh(oapiLoadMeshGlobal(BuildFilename("Right", nIndex)), &vecSlotOffset);
			dMass += TANK_MASS;
			vecCrsSec = CRSSEC_TANK;
			break;
		case CARGO:
			vecSlotOffset = _V(-CARGOSLOTOFFSET_X, 0.0, CARGOSLOTOFFSET_Z[nIndex]);
			pObj->AddMesh(oapiLoadMeshGlobal(BuildFilename("Left", nIndex)), &vecSlotOffset);
			vecSlotOffset.x = CARGOSLOTOFFSET_X;
			pObj->AddMesh(oapiLoadMeshGlobal(BuildFilename("Right", nIndex)), &vecSlotOffset);
			dMass += CARGO_MASS + m_arrdCargoWeights[nIndex];
			vecCrsSec = CRSSEC_CARGO;
			break;
		}
		dCrsSecZ = max(dCrsSecZ, vecCrsSec.z);
		dCrsSecX += vecCrsSec.x;
		dCrsSecY += vecCrsSec.y;
	}

	pObj->SetCrossSections(_V(46+dCrsSecX, 65+dCrsSecY, 30+dCrsSecZ));
	pObj->SetEmptyMass(dMass);

	double dConMass, dDisconMass;
	m_FuelManage.GetBrokenFuelMasses(dConMass, dDisconMass);

	PROPELLANT_HANDLE phMain = pObj->CreatePropellantResource(dConMass);
	pObj->CreatePropellantResource(dDisconMass);

	THRUSTER_HANDLE th;

	// main thrusters
	if(m_FuelManage.HasTankConnectionRearPart(th_main[0]))
	{
		th = pObj->CreateThruster (_V(-1.8,0,-16.15), _V(0,0,1), MAX_MAIN_THRUST, phMain, ISP);
		pObj->AddExhaust (th, 12, 1);
		pObj->SetThrusterLevel(th, GetThrusterLevel(th_main[0]));
	}

	if(m_FuelManage.HasTankConnectionRearPart(th_main[1]))
	{
		th = pObj->CreateThruster (_V( 1.8,0,-16.15), _V(0,0,1), MAX_MAIN_THRUST, phMain, ISP);
		pObj->AddExhaust (th, 12, 1);
		pObj->SetThrusterLevel(th, GetThrusterLevel(th_main[1]));
	}
	
	// hover thruster
	if(m_FuelManage.HasTankConnectionRearPart(th_hover[1]))
	{
		th = pObj->CreateThruster (_V(0,-2.4,-12.83), _V(0,1,0), MAX_HOVER_THRUST, phMain, ISP);
		pObj->AddExhaust (th, 10, 1);
		pObj->SetThrusterLevel(th, GetThrusterLevel(th_hover[1]));
	}

	// rotateable aux thrusters 

	if(m_FuelManage.HasTankConnectionRearPart(th_pod[0]))
	{
		th = pObj->CreateThruster (m_arrpAuxThrusters[0]->GetPos(), m_arrpAuxThrusters[0]->GetThrustDirVec(), GetThrusterMax0(th_pod[0]), phMain, ISP);
		if(m_bAuxPodDmg)
			pObj->AddExhaust (th, 4.5, 1.0, 1.6);
		else
			pObj->AddExhaust (th, 6, 0.7, 1.5);
		pObj->SetThrusterLevel(th, GetThrusterLevel(th_pod[0]));
	}

	if(m_FuelManage.HasTankConnectionRearPart(th_pod[1]))
	{
		th = pObj->CreateThruster (m_arrpAuxThrusters[1]->GetPos(), m_arrpAuxThrusters[1]->GetThrustDirVec(), GetThrusterMax0(th_pod[1]), phMain, ISP);
		pObj->AddExhaust (th, 6, 0.7, 1.5);
		pObj->SetThrusterLevel(th, GetThrusterLevel(th_pod[1]));
	}

	UINT uiMesh;

	if(m_bAuxPodDmg)
		uiMesh = pObj->AddMesh(oapiLoadMeshGlobal(BuildFilename("ThrDmg")));
	else
		uiMesh = pObj->AddMesh(oapiLoadMeshGlobal(BuildFilename("Thr")));
	
	static ANIMCOMP anmcLeft, anmcRight;
	static UINT LeftPodGrp[4] = {MSHGRP_AuxThrusterHousingLeft,MSHGRP_AuxThrusterLeft,MSHGRP_AuxThrusterLitLeft,MSHGRP_AuxThrusterGridLeft};
	static UINT RightPodGrp[4] = {25,26,27,32};

	anmcLeft.grp = LeftPodGrp;
	anmcLeft.ngrp = 4;
	anmcLeft.state0 = 0.0;
	anmcLeft.state1 = 1.0;
	anmcLeft.trans.P.rotparam.ref = _V(0,0,0);
	anmcLeft.trans.P.rotparam.axis = _V(1,0,0);
	anmcLeft.trans.P.rotparam.angle = (float) m_arrpAuxThrusters[0]->GetThrustDir();
	anmcLeft.trans.nmesh = uiMesh;
	anmcLeft.trans.ngrp = 0;
	anmcLeft.trans.transform = MESHGROUP_TRANSFORM::ROTATE;

	anmcRight.grp = RightPodGrp;
	anmcRight.ngrp = 4;
	anmcRight.state0 = 0.0;
	anmcRight.state1 = 1.0;
	anmcRight.trans.P.rotparam.ref = _V(0,0,0);
	anmcRight.trans.P.rotparam.axis = _V(1,0,0);
	anmcRight.trans.P.rotparam.angle = (float) m_arrpAuxThrusters[1]->GetThrustDir();
	anmcRight.trans.nmesh = 0;
	anmcRight.trans.ngrp = 0;
	anmcRight.trans.transform = MESHGROUP_TRANSFORM::ROTATE;

	UINT uiAnimSeq = pObj->RegisterAnimSequence(0.0);
	pObj->AddAnimComp(uiAnimSeq, &anmcRight);
	pObj->AddAnimComp(uiAnimSeq, &anmcLeft);
	pObj->SetAnimState(uiAnimSeq, 1.0);

}



void ShuttleA::SetRadDmgAlarm(bool bAlarm)
{
	m_bRadDmgAlarm = bAlarm;
}

void ShuttleA::AddParticleStreams()
{
	for(int i=0;i<2;i++)
	{
		if(m_hpsEmissHover[i] != NULL)
		{
			DelExhaustStream(m_hpsEmissHover[i]);
			m_hpsEmissHover[i] = NULL;
		}
		if(m_hpsDiffHover[i] != NULL)
		{
			DelExhaustStream(m_hpsDiffHover[i]);
			m_hpsDiffHover[i] = NULL;
		}
		if(m_hpsEmissAux[i] != NULL)
		{
			DelExhaustStream(m_hpsEmissAux[i]);
			m_hpsEmissAux[i] = NULL;
		}
		if(m_hpsDiffAux[i] != NULL)
		{
			DelExhaustStream(m_hpsDiffAux[i]);
			m_hpsDiffAux[i] = NULL;
		}
		if(m_hpsEmissMain[i] != NULL)
		{
			DelExhaustStream(m_hpsEmissMain[i]);
			m_hpsEmissMain[i] = NULL;
		}
		if(m_hpsDiffMain[i] != NULL)
		{
			DelExhaustStream(m_hpsDiffMain[i]);
			m_hpsDiffMain[i] = NULL;
		}
	}

	
	static PARTICLESTREAMSPEC pssEmissHover;
	pssEmissHover.flags = 0;
	pssEmissHover.srcsize = 1;
	pssEmissHover.srcrate = 100;
	pssEmissHover.v0 = 80;
	pssEmissHover.srcspread = 0.1;
	pssEmissHover.lifetime = 0.05;
	pssEmissHover.growthrate = 5.0;
	pssEmissHover.atmslowdown = 0.1;
	pssEmissHover.ltype = PARTICLESTREAMSPEC::EMISSIVE;
	pssEmissHover.levelmap = PARTICLESTREAMSPEC::LVL_PLIN;
	pssEmissHover.lmin = 0;
	pssEmissHover.lmax = 1;
	pssEmissHover.atmsmap = PARTICLESTREAMSPEC::ATM_PLIN;
	pssEmissHover.amin = 0.0;
	pssEmissHover.amax = 1.3;
	pssEmissHover.tex = m_hExhPtclTex;

	m_hpsEmissHover[0] = AddExhaustStream (th_hover[0], &pssEmissHover);
	m_hpsEmissHover[1] = AddExhaustStream (th_hover[1], &pssEmissHover);

	static PARTICLESTREAMSPEC pssDiffHover;
	pssDiffHover.flags = 0;
	pssDiffHover.srcsize = 2;
	pssDiffHover.srcrate = 100;
	pssDiffHover.v0 = 50;
	pssDiffHover.srcspread = 0.3;
	pssDiffHover.lifetime = 0.5;
	pssDiffHover.growthrate = 4.0;
	pssDiffHover.atmslowdown = 2;
	pssDiffHover.ltype = PARTICLESTREAMSPEC::DIFFUSE;
	pssDiffHover.levelmap = PARTICLESTREAMSPEC::LVL_PLIN;
	pssDiffHover.lmin = 0.0001;
	pssDiffHover.lmax = 2;
	pssDiffHover.atmsmap = PARTICLESTREAMSPEC::ATM_PLIN;
	pssDiffHover.amin = 0.000001;
	pssDiffHover.amax = 1.3;

	m_hpsDiffHover[0] = AddExhaustStream (th_hover[0], &pssDiffHover);
	m_hpsDiffHover[1] = AddExhaustStream (th_hover[1], &pssDiffHover);

	static PARTICLESTREAMSPEC pssEmissAux;
	pssEmissAux.flags = 0;
	pssEmissAux.srcsize = 0.7;
	pssEmissAux.srcrate = 100;
	pssEmissAux.v0 = 80;
	pssEmissAux.srcspread = 0.1;
	pssEmissAux.lifetime = 0.03;
	pssEmissAux.growthrate = 3.0;
	pssEmissAux.atmslowdown = 0.1;
	pssEmissAux.ltype = PARTICLESTREAMSPEC::EMISSIVE;
	pssEmissAux.levelmap = PARTICLESTREAMSPEC::LVL_PLIN;
	pssEmissAux.lmin = 0;
	pssEmissAux.lmax = 1;
	pssEmissAux.atmsmap = PARTICLESTREAMSPEC::ATM_PLIN;
	pssEmissAux.amin = 0.0;
	pssEmissAux.amax = 1.3;
	pssEmissAux.tex = m_hExhPtclTex;

	m_hpsEmissAux[0] = AddExhaustStream (th_pod[0], &pssEmissAux);
	m_hpsEmissAux[1] = AddExhaustStream (th_pod[1], &pssEmissAux);

	static PARTICLESTREAMSPEC pssDiffAux;
	pssDiffAux.flags = 0;
	pssDiffAux.srcsize = 1.4;
	pssDiffAux.srcrate = 100;
	pssDiffAux.v0 = 50;
	pssDiffAux.srcspread = 0.3;
	pssDiffAux.lifetime = 0.5;
	pssDiffAux.growthrate = 2.5;
	pssDiffAux.atmslowdown = 2;
	pssDiffAux.ltype = PARTICLESTREAMSPEC::DIFFUSE;
	pssDiffAux.levelmap = PARTICLESTREAMSPEC::LVL_PLIN;
	pssDiffAux.lmin = 0.0001;
	pssDiffAux.lmax = 2;
	pssDiffAux.atmsmap = PARTICLESTREAMSPEC::ATM_PLIN;
	pssDiffAux.amin = 0.000001;
	pssDiffAux.amax = 1.3;

	m_hpsDiffAux[0] = AddExhaustStream (th_pod[0], &pssDiffAux);
	m_hpsDiffAux[1] = AddExhaustStream (th_pod[1], &pssDiffAux);

	static PARTICLESTREAMSPEC pssEmissMain;
	pssEmissMain.flags = 0;
	pssEmissMain.srcsize = 1.0;
	pssEmissMain.srcrate = 100;
	pssEmissMain.v0 = 80;
	pssEmissMain.srcspread = 0.1;
	pssEmissMain.lifetime = 0.06;
	pssEmissMain.growthrate = 6.0;
	pssEmissMain.atmslowdown = 0.1;
	pssEmissMain.ltype = PARTICLESTREAMSPEC::EMISSIVE;
	pssEmissMain.levelmap = PARTICLESTREAMSPEC::LVL_PLIN;
	pssEmissMain.lmin = 0;
	pssEmissMain.lmax = 1;
	pssEmissMain.atmsmap = PARTICLESTREAMSPEC::ATM_PLIN;
	pssEmissMain.amin = 0.0;
	pssEmissMain.amax = 1.3;
	pssEmissMain.tex = m_hExhPtclTex;

	m_hpsEmissMain[0] = AddExhaustStream (th_main[0], &pssEmissMain);
	m_hpsEmissMain[1] = AddExhaustStream (th_main[1], &pssEmissMain);

	static PARTICLESTREAMSPEC pssDiffMain;
	pssDiffMain.flags = 0;
	pssDiffMain.srcsize = 2.0;
	pssDiffMain.srcrate = 100;
	pssDiffMain.v0 = 50;
	pssDiffMain.srcspread = 0.3;
	pssDiffMain.lifetime = 0.5;
	pssDiffMain.growthrate = 5;
	pssDiffMain.atmslowdown = 2;
	pssDiffMain.ltype = PARTICLESTREAMSPEC::DIFFUSE;
	pssDiffMain.levelmap = PARTICLESTREAMSPEC::LVL_PLIN;
	pssDiffMain.lmin = 0.0001;
	pssDiffMain.lmax = 2;
	pssDiffMain.atmsmap = PARTICLESTREAMSPEC::ATM_PLIN;
	pssDiffMain.amin = 0.000001;
	pssDiffMain.amax = 1.3;

	m_hpsDiffMain[0] = AddExhaustStream (th_main[0], &pssDiffMain);
	m_hpsDiffMain[1] = AddExhaustStream (th_main[1], &pssDiffMain);

}

bool ShuttleA::IsLanded()
{
	VESSELSTATUS2 vs;
	vs.version = 2;
	vs.flag = 0;
	GetStatusEx(&vs);
	return vs.status == 1;
}


void ShuttleA::CreateLights()
{
	m_thLights[LGT_RIGHT] = 
		CreateThruster (_V(7.75,0,0), _V(0, 1,0), 0.0, m_FuelManage.m_arrTanks[CFuelManagement::RCS].m_ph, ISP);
	AddExhaust (m_thLights[LGT_RIGHT], 0.0, 0.05, m_hLightGreenTex);

	m_thLights[LGT_LEFT] = 
		CreateThruster (_V(-7.75,0,0), _V(0, 1,0), 0.0, m_FuelManage.m_arrTanks[CFuelManagement::RCS].m_ph, ISP);
	AddExhaust (m_thLights[LGT_LEFT], 0.0, 0.05, m_hLightRedTex);

	m_thLights[LGT_AFT] = 
		CreateThruster (_V(0.0,1.71,-14.39), _V(0, 1,0), 0.0, m_FuelManage.m_arrTanks[CFuelManagement::RCS].m_ph, ISP);
	AddExhaust (m_thLights[LGT_AFT], 0.0, 0.05);

	m_thLights[LGT_RIGHT_STRB] = 
		CreateThruster (_V(7.75,0,0), _V(0, 1,0), 0.0, m_FuelManage.m_arrTanks[CFuelManagement::RCS].m_ph, ISP);
	AddExhaust (m_thLights[LGT_RIGHT_STRB], 0.0, 0.08);

	m_thLights[LGT_LEFT_STRB] = 
		CreateThruster (_V(-7.75,0,0), _V(0, 1,0), 0.0, m_FuelManage.m_arrTanks[CFuelManagement::RCS].m_ph, ISP);
	AddExhaust (m_thLights[LGT_LEFT_STRB], 0.0, 0.08);

	m_thLights[LGT_AFT_STRB] = 
		CreateThruster (_V(0,2.3,-12.0), _V(0, 1,0), 0.0, m_FuelManage.m_arrTanks[CFuelManagement::RCS].m_ph, ISP);
	AddExhaust (m_thLights[LGT_AFT_STRB], 0.0, 0.08);

	m_thLights[LGT_TOP_BCN] = 
		CreateThruster (_V(0,2.9,17.2), _V(0, 1,0), 0.0, m_FuelManage.m_arrTanks[CFuelManagement::RCS].m_ph, ISP);
	AddExhaust (m_thLights[LGT_TOP_BCN], 0.0, 0.08, m_hLightRedTex);

	m_thLights[LGT_BTM_BCN] = 
		CreateThruster (_V(0,-1.85,0), _V(0, 1,0), 0.0, m_FuelManage.m_arrTanks[CFuelManagement::RCS].m_ph, ISP);
	AddExhaust (m_thLights[LGT_BTM_BCN], 0.0, 0.08, m_hLightRedTex);
	
}

void ShuttleA::SetNavLight(bool bOn)
{
	m_bNavLight = bOn && !m_bShuttleBroken;
}

void ShuttleA::SetBcnLight(bool bOn)
{
	m_bBcnLight = bOn && !m_bShuttleBroken;
}

void ShuttleA::SetStrbLight(bool bOn)
{
	m_bStrbLight = bOn && !m_bShuttleBroken;
}


char* ShuttleA::BuildFilename(char *chEnd, int iCargoSlot)
{
	static char chFilename[256];

	if(iCargoSlot == -1)
	{
		if(strlen(m_chMeshShuttle) == 0)
			strcpy(chFilename, "ShA2");
		else
			strcpy(chFilename, m_chMeshShuttle);
	}
	else
	{
		if(strlen(m_arrMeshesCargoSlots[iCargoSlot]) == 0)
		{
			switch(m_arriCargoConf[iCargoSlot])
			{
			case EMPTY:
				strcpy(chFilename, "ShA2Stabilizer");
				break;
			case CARGO:
				strcpy(chFilename, "ShA2Cargo");
				break;
			case TANK:
				strcpy(chFilename, "ShA2Tank");
				break;
			}
		}
		else
			strcpy(chFilename, m_arrMeshesCargoSlots[iCargoSlot]);
	}
	strcat(chFilename, chEnd);
	return chFilename;
}


void ShuttleA::SetHullTempAlarm(bool bAlarm)
{
	m_bHullTempAlarm = bAlarm; 
}

int ShuttleA::ConsumeKey(char *keystate)
{
	if(m_bHandleHoverKeys)
	{
		if (KEYMOD_CONTROL (keystate)) 
		{
			if(KEYDOWN(keystate, OAPI_KEY_NUMPAD0)) 
			{
				IncThrusterGroupLevel(THGROUP_HOVER, oapiGetSysStep() * 0.2);
				m_dOldHoverLvl = GetThrusterGroupLevel(THGROUP_HOVER);
				RESETKEY(keystate, OAPI_KEY_NUMPAD0);
			}
			else if(KEYDOWN(keystate, OAPI_KEY_DECIMAL))
			{
				IncThrusterGroupLevel(THGROUP_HOVER, oapiGetSysStep() * -0.2);
				m_dOldHoverLvl = GetThrusterGroupLevel(THGROUP_HOVER);
				RESETKEY(keystate, OAPI_KEY_DECIMAL);
			}
		}
	}
	return 0;
}

// ==============================================================
// DLL entry point
// ==============================================================

BOOL WINAPI DllMain (HINSTANCE hModule,
					 DWORD ul_reason_for_call,
					 LPVOID lpReserved)
{
	int i;

	switch (ul_reason_for_call) {
	case DLL_PROCESS_ATTACH:
		g_Param.hDLL = hModule;
		g_Param.hFont[0] = CreateFont (-10, 0, 0, 0, 400, 0, 0, 0, 0, 0, 0, 0, 0, "Arial");
		g_Param.hPen[0] = CreatePen (PS_SOLID, 3, RGB (120,220,120));
		g_Param.hPen[1] = CreatePen (PS_SOLID, 3, RGB (0,150,0));
		g_Param.hPen[2] = CreatePen (PS_SOLID, 1, RGB (0,200,0));
		g_Param.hBrush[0] = CreateSolidBrush (RGB(0,128,0));
		g_Param.hBrush[1] = CreateSolidBrush (RGB(160,160,0));
		g_Param.hBrush[2] = CreateSolidBrush (RGB(190,0,0));
		g_Param.hBrush[3] = CreateSolidBrush (RGB(0,200,0));
		break;
	case DLL_PROCESS_DETACH:
		for (i = 0; i < 1; i++) DeleteObject (g_Param.hFont[i]);
		for (i = 0; i < 3; i++) DeleteObject (g_Param.hPen[i]);
		for (i = 0; i < 4; i++) DeleteObject (g_Param.hBrush[i]);
		break;
	}
	return TRUE;
}

// ==============================================================
// API interface
// ==============================================================

// Initialisation
DLLCLBK VESSEL *ovcInit (OBJHANDLE hvessel, int flightmodel)
{
	return new ShuttleA (hvessel, flightmodel);
}

// Cleanup
DLLCLBK void ovcExit (VESSEL *vessel)
{
	if (vessel) delete (ShuttleA*)vessel;
}

// Set vessel class caps
DLLCLBK void ovcSetClassCaps (VESSEL* vessel, FILEHANDLE cfg)
{
	((ShuttleA*)vessel)->SetClassCaps (cfg);
}

// Read status from scenario file
DLLCLBK void ovcLoadStateEx (VESSEL *vessel, FILEHANDLE scn, void *vs)
{
	((ShuttleA*)vessel)->LoadState (scn, vs);
}

// Save status to scenario file
DLLCLBK void ovcSaveState (VESSEL *vessel, FILEHANDLE scn)
{
	((ShuttleA*)vessel)->SaveState (scn);
}

DLLCLBK void ovcPostCreation (VESSEL *vessel)
{
	((ShuttleA*)vessel)->PostCreation();

}

DLLCLBK void ovcTimestep (VESSEL *vessel, double simt)
{
	((ShuttleA*)vessel)->Timestep (simt);
}

DLLCLBK bool ovcLoadPanel (VESSEL *vessel, int id)
{
	return ((ShuttleA*)vessel)->LoadPanel(id);		
}

DLLCLBK bool ovcPanelRedrawEvent (VESSEL *vessel, int id, int event, SURFHANDLE surf)
{
	return ((ShuttleA*)vessel)->RedrawPanel(id, event, surf);
}

DLLCLBK bool ovcPanelMouseEvent (VESSEL *vessel, int id, int event, int mx, int my)
{
	return ((ShuttleA*)vessel)->PanelMouseEvent(id, event, mx, my);
}

DLLCLBK void ovcMFDmode (VESSEL *vessel, int mfd, int mode)
{
	((ShuttleA*) vessel)->MFDmode(mfd, mode);
}

DLLCLBK void ovcNavmode (VESSEL *vessel, int mode, bool active)
{
	((ShuttleA*) vessel)->Navmode(mode, active);
}

DLLCLBK int ovcConsumeBufferedKey (VESSEL *vessel, DWORD key, bool down, char *kstate)
{
	return ((ShuttleA*)vessel)->ConsumeBufferedKey(key, down, kstate);
}

DLLCLBK int ovcConsumeKey (VESSEL *vessel, char *keystate)
{
	return ((ShuttleA*)vessel)->ConsumeKey(keystate);
}

DLLCLBK void ovcVisualCreated (VESSEL *vessel, VISHANDLE vis, int refcount)
{
	((ShuttleA*)vessel)->VisualCreated(vis);
}

DLLCLBK void ovcVisualDestroyed (VESSEL *vessel, VISHANDLE vis, int refcount)
{
	((ShuttleA*)vessel)->VisualDestroyed(vis);
}

DLLCLBK void ovcAnimate (VESSEL *vessel, double simt)
{
	((ShuttleA*)vessel)->Animate(simt);
}

bool GetBlinkStatus(double dFrequ, double dPulseWidth, double dOffset)
{
	double dDummy;
	return modf(oapiGetSimTime()*dFrequ-dOffset, &dDummy) < dPulseWidth;
}

