// UpperPanel.cpp: Implementation of Class CUpperPanel.
//
//////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <math.h>
#include "FuelManagement.h"
#include "UpperPanel.h"
#include "ShuttleA.h"
#include "resource.h"
#include "orbitersoundsdk.h"

extern GDIParams g_Param;

const WORD CUpperPanel::sc_pwBmpResIDs[NUMSURFACES] = {IDB_SWITCH2, IDB_SWITCH3, IDB_SWITCH4, 
IDB_INDICATOR1, IDB_INDICATOR2, IDB_TANKCONFIG, IDB_CARGOCONFIG, IDB_NO_TANK, 
IDB_OPENVALVE, IDB_RADAR_ON, IDB_RADAR_MALFUNCTION, IDB_FUELPUMP_LVL};

const int CUpperPanel::sc_arriThrustValvesY[9] = {11, 22, 69, 80, 126, 137, 
	0, 34, 91};

const double CUpperPanel::s_arrdRadarScales[NUM_RADAR_SCALES] = {10, 20, 50, 100, 200, 500, 1000, 2000, 5000, 10000, 20000, 50000, 100000, 200000, 500000, 1000000};

CUpperPanel::CUpperPanel(VESSEL* pVessel, int iPanelID) : CPanel(pVessel, iPanelID, NUM_AREANAMES, sc_pwBmpResIDs, NUMSURFACES)
{
	m_iRadarScaleIndex = 6;
}

CUpperPanel::~CUpperPanel()
{

}

bool CUpperPanel::Load()
{
	HBITMAP hBmp = LoadBitmap (g_Param.hDLL, MAKEINTRESOURCE (IDB_PANEL2));
	if(!hBmp)
		return false;

	CFuelManagement::STank *pTanks = ((ShuttleA*)m_pVessel)->m_FuelManage.m_arrTanks;

	oapiRegisterPanelBackground (hBmp, PANEL_ATTACH_TOP|PANEL_MOVEOUT_TOP, 0xffffff);
	oapiRegisterPanelArea (AreaID(AID_THR_FLOWLEVELS),    _R( 191, 102, 207, 257), PANEL_REDRAW_ALWAYS, PANEL_MOUSE_LBDOWN, PANEL_MAP_BACKGROUND);
	oapiRegisterPanelArea (AreaID(AID_THRGRP_VALVES),     _R( 247, 143, 257, 244), PANEL_REDRAW_ALWAYS, PANEL_MOUSE_LBDOWN, PANEL_MAP_BACKGROUND);
	oapiRegisterPanelArea (AreaID(AID_EXT_TANK_VLVFLOW1), _R( 313,  69, 351,  96), PANEL_REDRAW_ALWAYS, PANEL_MOUSE_LBDOWN, PANEL_MAP_BACKGROUND);
	oapiRegisterPanelArea (AreaID(AID_EXT_TANK_VLVFLOW2), _R( 313, 119, 351, 139), PANEL_REDRAW_ALWAYS, PANEL_MOUSE_LBDOWN, PANEL_MAP_BACKGROUND);
	oapiRegisterPanelArea (AreaID(AID_EXT_TANK_VLVFLOW3), _R( 313, 175, 351, 195), PANEL_REDRAW_ALWAYS, PANEL_MOUSE_LBDOWN, PANEL_MAP_BACKGROUND);
	oapiRegisterPanelArea (AreaID(AID_EXT_TANK_VLVFLOW4), _R( 313, 241, 351, 261), PANEL_REDRAW_ALWAYS, PANEL_MOUSE_LBDOWN, PANEL_MAP_BACKGROUND);
	oapiRegisterPanelArea (AreaID(AID_EXT_TANK1),         _R( 353,  38, 383,  88), pTanks[0].m_bPresent?PANEL_REDRAW_ALWAYS:PANEL_REDRAW_INIT, pTanks[0].m_bPresent?PANEL_MOUSE_LBDOWN:PANEL_MOUSE_IGNORE, PANEL_MAP_BACKGROUND);
	oapiRegisterPanelArea (AreaID(AID_EXT_TANK2),         _R( 353,  99, 383, 150), pTanks[1].m_bPresent?PANEL_REDRAW_ALWAYS:PANEL_REDRAW_INIT, pTanks[1].m_bPresent?PANEL_MOUSE_LBDOWN:PANEL_MOUSE_IGNORE, PANEL_MAP_BACKGROUND);
	oapiRegisterPanelArea (AreaID(AID_EXT_TANK3),         _R( 353, 160, 383, 210), pTanks[2].m_bPresent?PANEL_REDRAW_ALWAYS:PANEL_REDRAW_INIT, pTanks[2].m_bPresent?PANEL_MOUSE_LBDOWN:PANEL_MOUSE_IGNORE, PANEL_MAP_BACKGROUND);
	oapiRegisterPanelArea (AreaID(AID_EXT_TANK4),         _R( 353, 221, 383, 271), pTanks[3].m_bPresent?PANEL_REDRAW_ALWAYS:PANEL_REDRAW_INIT, pTanks[3].m_bPresent?PANEL_MOUSE_LBDOWN:PANEL_MOUSE_IGNORE, PANEL_MAP_BACKGROUND);
	oapiRegisterPanelArea (AreaID(AID_INT_TANK),          _R( 272, 210, 314, 267), PANEL_REDRAW_ALWAYS, PANEL_MOUSE_LBDOWN, PANEL_MAP_BACKGROUND);
	oapiRegisterPanelArea (AreaID(AID_RCS_TANK),          _R( 181,  65, 277,  90), PANEL_REDRAW_ALWAYS, PANEL_MOUSE_LBDOWN, PANEL_MAP_BACKGROUND);
	oapiRegisterPanelArea (AreaID(AID_FUEL_DOCKPORT),	  _R( 210,  31, 225,  50), PANEL_REDRAW_ALWAYS, PANEL_MOUSE_LBDOWN, PANEL_MAP_BACKGROUND);
	oapiRegisterPanelArea (AreaID(AID_FUEL_HATCHVENT),	  _R( 293,  41, 322,  51), PANEL_REDRAW_ALWAYS, PANEL_MOUSE_LBDOWN, PANEL_MAP_BACKGROUND);
	oapiRegisterPanelArea (AreaID(AID_FUEL_RCSPIPE),	  _R( 258,  99, 268, 109), PANEL_REDRAW_ALWAYS, PANEL_MOUSE_LBDOWN, PANEL_MAP_BACKGROUND);
	oapiRegisterPanelArea (AreaID(AID_FUELPUMP),		  _R( 269, 120, 307, 207), PANEL_REDRAW_ALWAYS, PANEL_MOUSE_LBDOWN|PANEL_MOUSE_LBPRESSED, PANEL_MAP_BACKGROUND);
	oapiRegisterPanelArea (AreaID(AID_AIRLOCK1SWITCH),    _R( 548,  38, 572,  88), PANEL_REDRAW_USER, PANEL_MOUSE_LBDOWN);
	oapiRegisterPanelArea (AreaID(AID_AIRLOCK1INDICATOR), _R( 544,  91, 578,  99), PANEL_REDRAW_USER, PANEL_MOUSE_IGNORE, PANEL_MAP_BACKGROUND);
	oapiRegisterPanelArea (AreaID(AID_DOCKSWITCH),        _R( 550, 128, 574, 178), PANEL_REDRAW_USER, PANEL_MOUSE_LBDOWN);
	oapiRegisterPanelArea (AreaID(AID_DOCKINDICATOR),     _R( 546, 180, 580, 188), PANEL_REDRAW_USER, PANEL_MOUSE_IGNORE, PANEL_MAP_BACKGROUND);
	oapiRegisterPanelArea (AreaID(AID_NAVLIGHT_SWITCH),   _R( 713,  38, 737,  88), PANEL_REDRAW_USER, PANEL_MOUSE_LBDOWN);
	oapiRegisterPanelArea (AreaID(AID_BCNLIGHT_SWITCH),   _R( 761,  38, 785,  88), PANEL_REDRAW_USER, PANEL_MOUSE_LBDOWN);
	oapiRegisterPanelArea (AreaID(AID_STRBLIGHT_SWITCH),  _R( 808,  38, 832,  88), PANEL_REDRAW_USER, PANEL_MOUSE_LBDOWN);
	oapiRegisterPanelArea (AreaID(AID_SHIPSCHEME),		  _R(  83,  29, 163, 249), PANEL_REDRAW_INIT, PANEL_MOUSE_IGNORE, PANEL_MAP_BACKGROUND);
	oapiRegisterPanelArea (AreaID(AID_RADAR),		      _R( 916,  19,1115, 218), PANEL_REDRAW_ALWAYS, PANEL_MOUSE_LBDOWN, PANEL_MAP_BACKGROUND);
	oapiSetPanelNeighbours (-1,-1,-1,0);
	return true;
}

void CUpperPanel::Init()
{
}

bool CUpperPanel::MouseEvent(int iID, int iEvent, int iX, int iY)
{
	ShuttleA* pSh = (ShuttleA*) m_pVessel;
	int iTank, iValve;
	int iIndex = AreaIndex(iID);
	if(iIndex < 0)
		return false;

	switch(iIndex)
	{
	case AID_AIRLOCK1SWITCH:
		pSh->ActivateAirlock (iY < 25 ? ShuttleA::DOOR_CLOSING : ShuttleA::DOOR_OPENING);
		return true;
	case AID_DOCKSWITCH:
		pSh->ActivateDockingPort (iY < 25 ? ShuttleA::DOOR_CLOSING : ShuttleA::DOOR_OPENING);
		return true;
	case AID_NAVLIGHT_SWITCH:
		pSh->SetNavLight(iY < 25);
		TriggerRedrawArea(AID_NAVLIGHT_SWITCH);
		return true;
	case AID_BCNLIGHT_SWITCH:
		pSh->SetBcnLight(iY < 25);
		TriggerRedrawArea(AID_BCNLIGHT_SWITCH);
		return true;
	case AID_STRBLIGHT_SWITCH:
		pSh->SetStrbLight(iY < 25);
		TriggerRedrawArea(AID_STRBLIGHT_SWITCH);
		return true;
	case AID_FUELPUMP:
		if(iX >= 4 && iX <= 34 && iY >= 5)
		{
			pSh->m_FuelManage.m_FuelPump.m_fTargetLvl = max(0.0, (((float)(67 - (iY - 5))) / 67.0) * 1.2);
		}
		return true;
	case AID_FUEL_DOCKPORT:
		if(iX < 10 && iY < 10)
		{
			PlayVesselWave(SND_LCD_BEEP,NOLOOP);
			pSh->m_FuelManage.ToggleValve(CFuelManagement::VLV_DOCK);
			TriggerRedrawArea(AID_FUEL_DOCKPORT);
			return true;
		}
		return false;
	case AID_FUEL_HATCHVENT:
		if(iX < 10 && iY < 10)
		{
			PlayVesselWave(SND_LCD_BEEP,NOLOOP);
			pSh->m_FuelManage.ToggleValve(CFuelManagement::VLV_HATCHVENT);
			TriggerRedrawArea(AID_FUEL_HATCHVENT);
			return true;
		}
		return false;
	case AID_FUEL_RCSPIPE:
		PlayVesselWave(SND_LCD_BEEP,NOLOOP);
		pSh->m_FuelManage.ToggleValve(CFuelManagement::VLV_RCSPIPE);
		TriggerRedrawArea(AID_FUEL_RCSPIPE);
		return true;
	case AID_THR_FLOWLEVELS:
		if(iX >= 2 && iX <= 11)
		{
			for(iValve = 0;iValve < 6;iValve++)
			{
				if(iY >= sc_arriThrustValvesY[iValve] && iY <= (sc_arriThrustValvesY[iValve] + 9))
				{
					PlayVesselWave(SND_LCD_BEEP,NOLOOP);
					pSh->m_FuelManage.ToggleValve(CFuelManagement::VLV_FRONT_HOVER + iValve);
					return true;
				}
			}
		}
		return false;
	case AID_THRGRP_VALVES:
		if(iX >= 0 && iX <= 9)
		{
			for(iValve = 0;iValve < 3;iValve++)
			{
				if(iY >= sc_arriThrustValvesY[iValve+6] && iY <= (sc_arriThrustValvesY[iValve+6] + 9))
				{
					PlayVesselWave(SND_LCD_BEEP,NOLOOP);
					((ShuttleA*)m_pVessel)->m_FuelManage.ToggleValve(CFuelManagement::VLV_HOVER + iValve);
					TriggerRedrawArea(AID_THRGRP_VALVES);
					return true;
				}
			}
		}
		return false;
	case AID_EXT_TANK_VLVFLOW1:
	case AID_EXT_TANK_VLVFLOW2:
	case AID_EXT_TANK_VLVFLOW3:
	case AID_EXT_TANK_VLVFLOW4:
		iTank = iIndex - AID_EXT_TANK_VLVFLOW1;
		if(iX >= 19 && iX <= 28 && iY >= 0 && iY <= 9)
		{
			PlayVesselWave(SND_LCD_BEEP,NOLOOP);
			((ShuttleA*)m_pVessel)->m_FuelManage.ToggleValve(iTank + CFuelManagement::VLV_EXT1);
			TriggerRedrawArea(AID_THRGRP_VALVES);
			return true;
		}
		else
			return false;
	case AID_INT_TANK:
		if(iX >= 6 && iX <= 15 && iY >= 1 && iY <= 10)
		{
			PlayVesselWave(SND_LCD_BEEP,NOLOOP);
			((ShuttleA*)m_pVessel)->m_FuelManage.ToggleValve(CFuelManagement::VLV_INT);
			TriggerRedrawArea(AID_THRGRP_VALVES);
			return true;
		}
		else
			return false;
	case AID_RCS_TANK:
		if(iX >= 12 && iX <= 21 && iY >= 7 && iY <= 16)
		{
			PlayVesselWave(SND_LCD_BEEP,NOLOOP);
			((ShuttleA*)m_pVessel)->m_FuelManage.ToggleValve(CFuelManagement::VLV_RCS);
			return true;
		}
		else
			return false;
	case AID_RADAR:
		if(iY <= 21)
		{
			if(iY < 1)
				return false;

			if(iX >= 1 && iX <= 20)
			{
				if(m_iRadarScaleIndex > 0)
					m_iRadarScaleIndex--;
				PlayVesselWave(SND_LCD_BEEP,NOLOOP);
				return true;
			}
			else if (iX >= 22 && iX <= 41)
			{
				if(m_iRadarScaleIndex+1 < NUM_RADAR_SCALES)
					m_iRadarScaleIndex++;
				PlayVesselWave(SND_LCD_BEEP,NOLOOP);
				return true;
			}
		}
		else if(iY >= 179)
		{
			if(iX >= 1 && iX <= 20)
			{
				pSh->m_pRadar->ToggleActive();
				PlayVesselWave(SND_LCD_BEEP,NOLOOP);
				return true;
			}
		}
		else if(pSh->m_pRadar->IsWrecked() && pSh->m_bRadDmgAlarm)
		{
			pSh->SetRadDmgAlarm(false);
			return true;
		}
		return false;
	}
	return false;
}

bool CUpperPanel::Redraw(int iID, int iEvent, SURFHANDLE sh)
{
	ShuttleA* pSh = (ShuttleA*) m_pVessel;
	int iCargoIndex, iIndex = AreaIndex(iID);
	int arriYPositions[4] = {52, 87, 142, 177};
	if(iIndex < 0)
		return false;

	switch(iIndex)
	{
	case AID_THR_FLOWLEVELS:
	case AID_THRGRP_VALVES:
	case AID_EXT_TANK_VLVFLOW1:
	case AID_EXT_TANK_VLVFLOW2:
	case AID_EXT_TANK_VLVFLOW3:
	case AID_EXT_TANK_VLVFLOW4:
	case AID_EXT_TANK1:
	case AID_EXT_TANK2:
	case AID_EXT_TANK3:
	case AID_EXT_TANK4:
	case AID_INT_TANK:
	case AID_RCS_TANK:
	case AID_FUEL_DOCKPORT:
	case AID_FUEL_HATCHVENT:
	case AID_FUEL_RCSPIPE:
	case AID_FUELPUMP:
		RedrawPanel_Fuelstatus (sh, iIndex);
		return true;
	case AID_AIRLOCK1SWITCH:
		oapiBlt (sh, SurfHandle(SURF_SWITCH2), 0, 0, pSh->lock_status == ShuttleA::DOOR_OPEN ||
			pSh->lock_status == ShuttleA::DOOR_OPENING ? 24:0, 0, 24, 50);
		return true;
	case AID_AIRLOCK1INDICATOR:
		switch (pSh->lock_status) {
		case ShuttleA::DOOR_CLOSED: oapiBlt (sh, SurfHandle(SURF_INDICATOR1), 0, 0, 0,  0, 34, 8); break;
		case ShuttleA::DOOR_OPEN:   oapiBlt (sh, SurfHandle(SURF_INDICATOR1), 0, 0, 0, 16, 34, 8); break;
		default:                    oapiBlt (sh, SurfHandle(SURF_INDICATOR1), 0, 0, 0,  8, 34, 8); break;
		}
		return true;
	case AID_DOCKSWITCH:
		oapiBlt (sh, SurfHandle(SURF_SWITCH4), 0, 0, pSh->dock_status == ShuttleA::DOOR_OPEN ||
			pSh->dock_status == ShuttleA::DOOR_OPENING ? 24:0, 0, 24, 50);
		return true;
	case AID_DOCKINDICATOR:
		switch (pSh->dock_status) {
		case ShuttleA::DOOR_CLOSED: oapiBlt (sh, SurfHandle(SURF_INDICATOR2), 0, 0, 0,  0, 34, 8); break;
		case ShuttleA::DOOR_OPEN:   oapiBlt (sh, SurfHandle(SURF_INDICATOR2), 0, 0, 0, 16, 34, 8); break;
		default:                    oapiBlt (sh, SurfHandle(SURF_INDICATOR2), 0, 0, 0,  8, 34, 8); break;
		}
		return true;
	case AID_NAVLIGHT_SWITCH:
		if(pSh->m_bNavLight)
			oapiBlt (sh, SurfHandle(SURF_SWITCH3), 0, 0, 0, 0, 24, 50);
		else
			oapiBlt (sh, SurfHandle(SURF_SWITCH3), 0, 0, 24, 0, 24, 50);
		return true;
	case AID_BCNLIGHT_SWITCH:
		if(pSh->m_bBcnLight)
			oapiBlt (sh, SurfHandle(SURF_SWITCH3), 0, 0, 0, 0, 24, 50);
		else
			oapiBlt (sh, SurfHandle(SURF_SWITCH3), 0, 0, 24, 0, 24, 50);
		return true;
	case AID_STRBLIGHT_SWITCH:
		if(pSh->m_bStrbLight)
			oapiBlt (sh, SurfHandle(SURF_SWITCH3), 0, 0, 0, 0, 24, 50);
		else
			oapiBlt (sh, SurfHandle(SURF_SWITCH3), 0, 0, 24, 0, 24, 50);
		return true;
	case AID_SHIPSCHEME:
		for(iCargoIndex = 0;iCargoIndex < 4;iCargoIndex++)
		{
			switch(pSh->m_arriCargoConf[iCargoIndex])
			{
			case ShuttleA::TANK:
				oapiBlt (sh, SurfHandle(SURF_TANKCONFIG), 3, arriYPositions[iCargoIndex], 3, arriYPositions[iCargoIndex], 67, 35);
				break;
			case ShuttleA::CARGO:
				oapiBlt (sh, SurfHandle(SURF_CARGOCONFIG), 3, arriYPositions[iCargoIndex], 3, arriYPositions[iCargoIndex], 67, 35);
				break;
			}
		}
		return true;
	case AID_RADAR:
		RedrawPanel_Radar(sh);
		return true;
	}

	return false;
}

void CUpperPanel::RedrawPanel_Fuelstatus (SURFHANDLE sh, int iPart)
{
	char cbuf[20];
	double dCurPropMass;
	int iTank, iValve;
	bool bEarlyDCRelease = false;
	int iValveState;
	int iY;
	CFuelManagement::STank* pTank;

	ShuttleA* pSh = (ShuttleA*) m_pVessel;
	CFuelManagement* pFM = &pSh->m_FuelManage;

	HDC hDC = oapiGetDC (sh);
	SelectObject (hDC, g_Param.hFont[0]);
	SetTextColor (hDC, RGB(224,224,224));
	SetBkMode (hDC, TRANSPARENT);

	switch (iPart) {
	case AID_THR_FLOWLEVELS:
		sprintf (cbuf, "%0.1f", m_pVessel->GetThrusterLevel (((ShuttleA*)m_pVessel)->th_hover[0]) * MAX_HOVER_THRUST / ISP);
		TextOut (hDC, 0, 0, cbuf, strlen (cbuf));
		sprintf (cbuf, "%0.1f", m_pVessel->GetThrusterLevel (((ShuttleA*)m_pVessel)->th_hover[1]) * MAX_HOVER_THRUST / ISP);
		TextOut (hDC, 0, 30, cbuf, strlen (cbuf));
		sprintf (cbuf, "%0.1f", m_pVessel->GetThrusterLevel (((ShuttleA*)m_pVessel)->th_pod[1]) * MAX_RETRO_THRUST / ISP);
		TextOut (hDC, 0, 58, cbuf, strlen (cbuf));
		sprintf (cbuf, "%0.1f", m_pVessel->GetThrusterLevel (((ShuttleA*)m_pVessel)->th_pod[0]) * pSh->GetThrusterMax (pSh->th_pod[0]) / ISP);
		TextOut (hDC, 0, 88, cbuf, strlen (cbuf));
		sprintf (cbuf, "%0.1f", m_pVessel->GetThrusterLevel (((ShuttleA*)m_pVessel)->th_main[0]) * MAX_MAIN_THRUST / ISP);
		TextOut (hDC, 0, 115, cbuf, strlen (cbuf));
		sprintf (cbuf, "%0.1f", m_pVessel->GetThrusterLevel (((ShuttleA*)m_pVessel)->th_main[1]) * MAX_MAIN_THRUST / ISP);
		TextOut (hDC, 0, 145, cbuf, 3);
		for(iValve = 0;iValve < 6;iValve++)
		{
			if(pFM->GetValveState(CFuelManagement::VLV_FRONT_HOVER + iValve) != CFuelManagement::VLVST_OFF)
			{
				if(!bEarlyDCRelease)
				{
					bEarlyDCRelease = true;
					oapiReleaseDC (sh, hDC);
				}
				oapiBlt(sh, SurfHandle(SURF_OPENVALVE), 2, sc_arriThrustValvesY[iValve], 0, 0, 10, 10);
			}
		}
		break;
	case AID_THRGRP_VALVES:
		for(iValve = 0;iValve < 3;iValve++)
		{
			iValveState = pFM->GetValveState(CFuelManagement::VLV_HOVER + iValve);
			if(iValveState != CFuelManagement::VLVST_OFF)
			{
				if(!bEarlyDCRelease)
				{
					bEarlyDCRelease = true;
					oapiReleaseDC (sh, hDC);
				}
				oapiBlt(sh, SurfHandle(SURF_OPENVALVE), 0, sc_arriThrustValvesY[iValve+6], 0, 0, 10, 10);
			}
		}
		break;
	case AID_EXT_TANK_VLVFLOW1:
	case AID_EXT_TANK_VLVFLOW2:
	case AID_EXT_TANK_VLVFLOW3:
	case AID_EXT_TANK_VLVFLOW4:
		iTank = iPart - AID_EXT_TANK_VLVFLOW1;
		pTank = &pFM->m_arrTanks[iTank+CFuelManagement::EXT1];
		iValveState = pFM->GetValveState(iTank + CFuelManagement::VLV_EXT1);
		sprintf (cbuf, "%0.1f", pTank->m_dCurFlowRate);
		TextOut (hDC, 13, 8, cbuf, strlen(cbuf));
		if(iValveState != CFuelManagement::VLVST_OFF)
		{
			bEarlyDCRelease = true;
			oapiReleaseDC (sh, hDC);
			
			oapiBlt(sh, SurfHandle(SURF_OPENVALVE), 19, 0, 
				iValveState==CFuelManagement::VLVST_SRC?
				0:iValveState==CFuelManagement::VLVST_DST?10:40, 0, 10, 10);
		}
		break;
	case AID_EXT_TANK1:
	case AID_EXT_TANK2:
	case AID_EXT_TANK3:
	case AID_EXT_TANK4:
		iTank = iPart - AID_EXT_TANK1;
		pTank = &pFM->m_arrTanks[iTank+CFuelManagement::EXT1];
		SelectObject (hDC, g_Param.hBrush[0]);
		if(pTank->m_bPresent)
		{
			dCurPropMass = pTank->m_dCurFuel;
			Rectangle (hDC,  0, 50, 30, (int)((1.0-(dCurPropMass/pTank->m_dMaxFuel))*50.0));
			sprintf (cbuf, "%0.0f", dCurPropMass);
			SetTextAlign (hDC, TA_CENTER);
			TextOut (hDC, 15, 20, cbuf, strlen (cbuf));

		}
		else
		{
			// release the DC here, since we want to Blit
			bEarlyDCRelease = true;
			oapiReleaseDC (sh, hDC);
			oapiBlt(sh, SurfHandle(SURF_NO_TANK), 0, 0, 0, 0, 30, 50, 0xffffff);
		}
		break;
	case AID_INT_TANK:
		pTank = &pFM->m_arrTanks[CFuelManagement::INT];
		iValveState = pFM->GetValveState(CFuelManagement::VLV_INT);
		SelectObject (hDC, g_Param.hBrush[0]);
		dCurPropMass = pTank->m_dCurFuel;
		Rectangle (hDC,  3, 58, 31, 20+(int)((1.0-(dCurPropMass / pTank->m_dMaxFuel))*38.0));
		sprintf (cbuf, "%0.1f", pTank->m_dCurFlowRate);
		TextOut (hDC, 19, 0, cbuf, strlen(cbuf));
		sprintf (cbuf, "%0.0f", dCurPropMass);
		SetTextAlign (hDC, TA_CENTER);
		TextOut (hDC, 17, 34, cbuf, strlen (cbuf));
		if(iValveState != CFuelManagement::VLVST_OFF)
		{
			bEarlyDCRelease = true;
			oapiReleaseDC (sh, hDC);
			oapiBlt(sh, SurfHandle(SURF_OPENVALVE), 6, 2, 
				iValveState==CFuelManagement::VLVST_SRC?20:
					iValveState==CFuelManagement::VLVST_DST?30:40, 0, 10, 10);
		}
		break;
	case AID_RCS_TANK:
		pTank = &pFM->m_arrTanks[CFuelManagement::RCS];
		iValveState = pFM->GetValveState(CFuelManagement::VLV_RCS);
		SelectObject (hDC, g_Param.hBrush[0]);
		dCurPropMass = pTank->m_dCurFuel;
		Rectangle (hDC, 68, 25, 96, (int)((1.0-(dCurPropMass / pTank->m_dMaxFuel))*25.0));
		sprintf (cbuf, "%0.2f", pTank->m_dCurFlowRate);
		TextOut (hDC, 4, 15, cbuf, strlen(cbuf));
		SetTextAlign (hDC, TA_CENTER);
		sprintf (cbuf, "%0.1f", dCurPropMass);
		TextOut (hDC, 83, 7, cbuf, strlen(cbuf));
		if(iValveState != CFuelManagement::VLVST_OFF)
		{
			bEarlyDCRelease = true;
			oapiReleaseDC (sh, hDC);
			oapiBlt(sh, SurfHandle(SURF_OPENVALVE), 12, 7, 0, 0, 10, 10);
		}
		break;
	case AID_FUEL_DOCKPORT:
		if(!pSh->DockingStatus(0))
		{
			bEarlyDCRelease = true;
			oapiReleaseDC (sh, hDC);
			oapiBlt(sh, SurfHandle(SURF_OPENVALVE), 0, 0, 40, 0, 10, 10);
		}
		else
		{
			iValveState = pFM->GetValveState(CFuelManagement::VLV_DOCK);
			if(iValveState != CFuelManagement::VLVST_OFF)
			{
				bEarlyDCRelease = true;
				oapiReleaseDC (sh, hDC);
				oapiBlt(sh, SurfHandle(SURF_OPENVALVE), 0, 0, iValveState==CFuelManagement::VLVST_DST?0:10, 0, 10, 10);
			}
		}
		break;
	case AID_FUEL_HATCHVENT:
		iValveState = pFM->GetValveState(CFuelManagement::VLV_HATCHVENT);
		if(iValveState != CFuelManagement::VLVST_OFF)
		{
			bEarlyDCRelease = true;
			oapiReleaseDC (sh, hDC);
			oapiBlt(sh, SurfHandle(SURF_OPENVALVE), 0, 0, iValveState==CFuelManagement::VLVST_DST?20:30, 0, 10, 10);
		}
		break;
	case AID_FUEL_RCSPIPE:
		iValveState = pFM->GetValveState(CFuelManagement::VLV_RCSPIPE);
		if(iValveState != CFuelManagement::VLVST_OFF)
		{
			bEarlyDCRelease = true;
			oapiReleaseDC (sh, hDC);
			oapiBlt(sh, SurfHandle(SURF_OPENVALVE), 0, 0, iValveState==CFuelManagement::VLVST_DST?20:30, 0, 10, 10);
		}
		break;
	case AID_FUELPUMP:
		sprintf (cbuf, "%0.0f", pFM->m_FuelPump.m_fCurFlowRate);
		TextOut (hDC, 4, 75, cbuf, strlen(cbuf));
		iY = (int) ((pFM->m_FuelPump.m_fActLvl / 1.2) * 65);
		bEarlyDCRelease = true;
		oapiReleaseDC (sh, hDC);
		oapiBlt(sh, SurfHandle(SURF_FUELPUMPLVL), 6, 73 - iY, 0, 66 - iY, 27, iY);
		iY = (int) ((pFM->m_FuelPump.m_fTargetLvl / 1.2) * 65);
		oapiBlt(sh, SurfHandle(SURF_FUELPUMPLVL), 2, 71 - iY, 27, 5, 3, 5);
		oapiBlt(sh, SurfHandle(SURF_FUELPUMPLVL), 34, 71 - iY, 27, 0, 3, 5);
		break;
	} 
	if(!bEarlyDCRelease)
		oapiReleaseDC (sh, hDC);
}

void CUpperPanel::RedrawPanel_Radar(SURFHANDLE sh)
{
	ShuttleA* pSh = (ShuttleA*) m_pVessel;
	double dX, dZ, dSizeHalf, dRadarScale ;
	int iXLeft, iYTop, iXRight, iYBottom;
	char chBuf[15];

	CRadar* pRadar = pSh->m_pRadar;

	if(pRadar->IsWrecked())
	{
		if(GetBlinkStatus(2.0))
			oapiBlt(sh, SurfHandle(SURF_RADAR_MALFUNCTION), 44, 89, 0, 0, 110, 20);
		return;
	}

	if(pRadar->IsActive())
	{
		oapiBlt(sh, SurfHandle(SURF_RADAR_ON), 1, 179, 0, 0, 20, 20);


		dRadarScale = s_arrdRadarScales[m_iRadarScaleIndex];

		HDC hDC = oapiGetDC (sh);
		SelectObject (hDC, g_Param.hPen[2]);	
		SelectObject (hDC, g_Param.hBrush[3]);

		MAP_OBJH2RadObj* pRadObjs = pSh->m_pRadar->GetRadObjs();
		MAP_OBJH2RadObj::iterator iterator;
		for(iterator = pRadObjs->begin();iterator != pRadObjs->end();iterator++)
		{
			if(iterator->second.m_bValid && iterator->second.m_dDistance3D < dRadarScale)
			{
				dX = -sin(iterator->second.m_dAngle2D) * iterator->second.m_dDistance2D;
				dZ = cos(iterator->second.m_dAngle2D) * iterator->second.m_dDistance2D;
				dSizeHalf = iterator->second.m_dSize;
				dSizeHalf = max(dSizeHalf/dRadarScale, 0.015) * dRadarScale;
				iXLeft = 100 + (int) (((dX - dSizeHalf) / dRadarScale) * 100 + 0.5);
				iXRight = 100 + (int) (((dX + dSizeHalf) / dRadarScale) * 100 + 0.5);
				iYTop = 100 + (int) (((-dZ - dSizeHalf) / dRadarScale) * 100 + 0.5);
				iYBottom = 100 + (int) (((-dZ + dSizeHalf) / dRadarScale) * 100 + 0.5);
				
				Ellipse(hDC, iXLeft, iYTop, iXRight, iYBottom);
			}
		}

		SelectObject (hDC, g_Param.hFont[0]);
		SetTextColor (hDC, RGB(0,150,0));
		SetBkMode (hDC, TRANSPARENT);
		SetTextAlign(hDC, TA_RIGHT);
		
		sprintf (chBuf, "%4.1f %s", dRadarScale > 999 ?dRadarScale/1000:dRadarScale, dRadarScale > 999 ?"km":"m");
		TextOut (hDC, 199, 188, chBuf, strlen (chBuf));

		oapiReleaseDC(sh, hDC);
	}
}

void CUpperPanel::SetRadarScaleIndex(int iRadarScaleIndex)
{
	if(iRadarScaleIndex >= 0 && iRadarScaleIndex < NUM_RADAR_SCALES)
		m_iRadarScaleIndex = iRadarScaleIndex;
}

